<?php

namespace App\Http\Controllers;

use App\Http\Resources\CaslPermissionResource;
use App\Http\Resources\User\UserAuthCheckResource;
use App\Http\Resources\UserAuthResource;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;

class AuthController extends Controller
{
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'surname' => 'required|string',
            'email' => 'required|string|email|unique:users',
            'password' => 'required|string|',
            'c_password'=>'required|same:password',
        ]);

        $user = new User([
            'name' => $request->name.' '.$request->surname,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'uniq_id' => Str::random(52),
        ]);

        $role = Role::where('name', 'member')->first();

        $user->assignRole('member');
        $user->givePermissionTo($role->permissions);

        if($user->save()){
            return response()->json([
                'message' => 'Successfully created user!'
            ], 201);
        }else{
            return response()->json(['error'=>'Provide proper details']);
        }
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|string|email',
            'password' => 'required|string',
            'remember_me' => 'boolean'
        ]);

        $credentials = request(['email', 'password']);
        if(!Auth::attempt($credentials))
        {
            return response()->json([
                'message' => 'Unauthorized'
            ], 401);
        }

        $user = User::where('email', $request->email)->first();
        $accessToken = $request->user()->createToken($user->name);
        $response = $this->generateLoginResponse($user, $accessToken->plainTextToken);

        return response()->json($response);
    }

    public function user(Request $request)
    {
        //$user = Auth::user();
        $user = auth('sanctum')->user();
        return response()->json(new UserAuthCheckResource($user));

    }

    public function logout(Request $request)
    {
        $request->user()->tokens()->delete();

        return response()->json([
            'message' => 'Successfully logged out'
        ]);
    }

    private function generateLoginResponse(User $user, string $accessToken): array
    {
        $abilities = CaslPermissionResource::collection($user->permissions);
        return [
            'message' => ('Logged in successfully!'),
            'user' => UserAuthResource::make($user)->additional(['abilities' => $abilities]),
            'userAbilities' => $abilities,
            'token' => $accessToken,
        ];
    }

    /**
     * Refresh user permissions for CASL ability system
     */
    public function refreshPermissions(Request $request): JsonResponse
    {
        $user = $request->user();
        $abilities = CaslPermissionResource::collection($user->permissions);
        
        return response()->json([
            'message' => 'Permissions refreshed successfully',
            'userAbilities' => $abilities,
        ]);
    }
}

