<?php

namespace App\Http\Controllers;

use App\Http\Resources\Category\EditResource;
use App\Http\Resources\Category\ListResource;
use App\Http\Resources\Category\VariantListResource;
use App\Http\Resources\Category\VariantResource;
use App\Models\Categories;
use App\Models\CategoryVariants;
use App\Models\Variant\Variants;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Turkey\Cities\Models\City;

class CategoryController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $perPage = $request->perPage ?? 10;

        $category = Categories::orderBy('id', 'DESC')->paginate($perPage);

        return response()->json([
            'pagination' => $category,
            'collection' => ListResource::collection($category)
        ]);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'status' => 'required',
            'parent_id' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'Error',
                'message' => $validator->errors()
            ], 400);
        }

        $category = new Categories();
        $category->name = $request->name;
        $category->icon = $request->icon ? $request->icon : null;
        $category->parent_id = $request->parent_id ? $request->parent_id : null;
        $category->slug = Str::slug($request->name);
        $category->status = $request->status;
        $category->populer_status = $request->populer_status;
        $category->save();

        $variantArr = json_decode($request->variants);
        if(count($variantArr) > 0){
            foreach ($variantArr as $item){
                $categoryVariant = new CategoryVariants();
                $categoryVariant->category_id = $category->id;
                $categoryVariant->variant_id = $item;
                $categoryVariant->save();
            }
        }

        return response()->json([
            'status' => 'Success',
            'message' => 'Kategori Başarılı Şekilde Eklendi'
        ]);
    }

    function categoryTree(): JsonResponse
    {
        $categories = Categories::orderBy('id', 'ASC')->get()->toTree();

        $formatted = $categories->map($recursive = function ($item) use (&$recursive) {
            $map = [
                'id' => $item->id,
                'label' => $item->name,
                "opened" => false,
                'icon' => $item->icon,
            ];

            if ($item->children->isNotEmpty()) {
                $map['children'] = $item->children->map($recursive);
            }

            return $map;
        });

        return response()->json($formatted, 200);
    }

    public function update(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'status' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'Error',
                'message' => $validator->errors()
            ], 400);
        }

        $category = Categories::find($request->id);
        $category->name = $request->name;
        $category->icon = $request->icon !== "null" ? $request->icon : null;
        $category->parent_id = $request->parent_id !== "0" ? $request->parent_id : 0;
        $category->slug = Str::slug($request->name);
        $category->status = $request->status;
        $category->populer_status = $request->populer_status;
        $category->save();

        if (count(json_decode($request->variants)) > 0){
            $category->variants()->delete();
            foreach (json_decode($request->variants) as $item){
                $categoryVariant = new CategoryVariants();
                $categoryVariant->category_id = $category->id;
                $categoryVariant->variant_id = $item;
                $categoryVariant->save();
            }
        }


        return response()->json([
            'status' => 'Success',
            'message' => 'Kategori Başarılı Şekilde Güncellendi'
        ]);
    }

    public function edit(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'Error',
                'message' => $validator->errors()
            ], 400);
        }

        $category = Categories::find($request->id);
        return response()->json(new EditResource($category));
    }

    public function destroy(Request $request): JsonResponse
    {

        $validator = Validator::make($request->all(), [
            'id' => 'required',
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'Error',
                'message' => $validator->errors()
            ], 400);
        }

        $category =  Categories::find($request->id);
        $category->variants()->delete();
        $category->delete();

        return response()->json([
            'status' => 'Success',
            'message' => 'Kategori Başarılı Şekilde Silindi'
        ]);
    }

    public function variants(Request $request): JsonResponse
    {
        $variants = Variants::all();
        return response()->json(VariantListResource::collection($variants));
    }
}
