<?php

namespace App\Http\Controllers;

use App\Models\PremiumPackage;
use App\Models\User;
use App\Models\UserPremium;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PremiumController extends Controller
{
    /**
     * Get premium users with filtering and pagination
     */
    public function getPremiumUsers(Request $request): JsonResponse
    {
        try {
            $query = UserPremium::with(['user:id,name,email,avatar', 'package:id,name,duration'])
                ->whereHas('user');

            // Search filter
            if ($request->filled('search')) {
                $search = $request->get('search');
                $query->whereHas('user', function ($q) use ($search) {
                    $q->where('name', 'LIKE', "%{$search}%")
                      ->orWhere('email', 'LIKE', "%{$search}%");
                });
            }

            // Status filter
            if ($request->filled('status')) {
                $status = $request->get('status');
                switch ($status) {
                    case 'active':
                        $query->where('status', 1)->where('end_date', '>', now());
                        break;
                    case 'expired':
                        $query->where('end_date', '<=', now())->where('status', 1);
                        break;
                    case 'cancelled':
                        $query->where('status', 0);
                        break;
                }
            }

            // Package filter
            if ($request->filled('package')) {
                $query->where('package_id', $request->get('package'));
            }

            // Pagination
            $perPage = $request->get('per_page', 10);
            $users = $query->orderBy('created_at', 'desc')->paginate($perPage);

            // Transform data to include status text and formatted dates
            $users->getCollection()->transform(function ($subscription) {
                return [
                    'id' => $subscription->id,
                    'user' => $subscription->user,
                    'package' => $subscription->package,
                    'status' => $subscription->status_text,
                    'started_at' => $subscription->start_date,
                    'expires_at' => $subscription->end_date,
                    'amount' => $subscription->price,
                    'created_at' => $subscription->created_at,
                ];
            });

            return response()->json([
                'data' => $users->items(),
                'total' => $users->total(),
                'per_page' => $users->perPage(),
                'current_page' => $users->currentPage(),
                'last_page' => $users->lastPage(),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Premium kullanıcılar getirilemedi.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Cancel user premium subscription
     */
    public function cancelSubscription(Request $request, $userId): JsonResponse
    {
        try {
            $subscription = UserPremium::where('user_id', $userId)
                ->where('status', 1)
                ->first();

            if (!$subscription) {
                return response()->json([
                    'message' => 'Aktif premium abonelik bulunamadı.'
                ], 404);
            }

            $subscription->update(['status' => 0]);

            return response()->json([
                'message' => 'Premium abonelik başarıyla iptal edildi.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Abonelik iptal edilirken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Extend user premium subscription
     */
    public function extendSubscription(Request $request, $userId): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'days' => 'required|integer|min:1|max:365'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'Geçersiz gün sayısı.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $subscription = UserPremium::where('user_id', $userId)
                ->where('status', 1)
                ->first();

            if (!$subscription) {
                return response()->json([
                    'message' => 'Aktif premium abonelik bulunamadı.'
                ], 404);
            }

            $days = $request->get('days');
            $currentEndDate = $subscription->end_date;
            
            // If expired, extend from now, otherwise extend from current end date
            $newEndDate = $currentEndDate->isFuture() 
                ? $currentEndDate->addDays($days)
                : now()->addDays($days);

            $subscription->update(['end_date' => $newEndDate]);

            return response()->json([
                'message' => "Premium abonelik {$days} gün uzatıldı.",
                'new_end_date' => $newEndDate
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Abonelik uzatılırken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all premium packages
     */
    public function getPackages(Request $request): JsonResponse
    {
        try {
            $query = PremiumPackage::query();

            // Search filter
            if ($request->filled('search')) {
                $search = $request->get('search');
                $query->where('name', 'LIKE', "%{$search}%")
                      ->orWhere('description', 'LIKE', "%{$search}%");
            }

            $packages = $query->ordered()->get();

            return response()->json([
                'data' => $packages
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Premium paketleri getirilemedi.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a new premium package
     */
    public function storePackage(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'duration' => 'required|string|max:255',
                'price' => 'required|numeric|min:0',
                'original_price' => 'nullable|numeric|min:0',
                'description' => 'nullable|string',
                'features' => 'nullable|array',
                'features.*' => 'string',
                'popular' => 'nullable|in:true,false,1,0',
                'color' => 'required|string|in:primary,success,warning,error,info',
                'is_active' => 'nullable|in:true,false,1,0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'Geçersiz veri.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();
            $data['popular'] = filter_var($request->get('popular'), FILTER_VALIDATE_BOOLEAN);
            $data['is_active'] = filter_var($request->get('is_active'), FILTER_VALIDATE_BOOLEAN);

            $package = PremiumPackage::create($data);

            return response()->json([
                'message' => 'Premium paket başarıyla oluşturuldu.',
                'data' => $package
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Paket oluşturulurken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a premium package
     */
    public function updatePackage(Request $request, $id): JsonResponse
    {
        try {
            $package = PremiumPackage::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'duration' => 'required|string|max:255',
                'price' => 'required|numeric|min:0',
                'original_price' => 'nullable|numeric|min:0',
                'description' => 'nullable|string',
                'features' => 'nullable|array',
                'features.*' => 'string',
                'popular' => 'nullable|in:true,false,1,0',
                'color' => 'required|string|in:primary,success,warning,error,info',
                'is_active' => 'nullable|in:true,false,1,0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'Geçersiz veri.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();
            $data['popular'] = filter_var($request->get('popular'), FILTER_VALIDATE_BOOLEAN);
            $data['is_active'] = filter_var($request->get('is_active'), FILTER_VALIDATE_BOOLEAN);

            $package->update($data);

            return response()->json([
                'message' => 'Premium paket başarıyla güncellendi.',
                'data' => $package
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Paket güncellenirken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a premium package
     */
    public function destroyPackage($id): JsonResponse
    {
        try {
            $package = PremiumPackage::findOrFail($id);

            // Check if package is used in any subscriptions
            $subscriptionCount = UserPremium::where('package_id', $id)->count();
            
            if ($subscriptionCount > 0) {
                return response()->json([
                    'message' => 'Bu paket kullanımda olduğu için silinemez. Önce pasif hale getirin.'
                ], 422);
            }

            $package->delete();

            return response()->json([
                'message' => 'Premium paket başarıyla silindi.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Paket silinirken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle package status
     */
    public function togglePackageStatus($id): JsonResponse
    {
        try {
            $package = PremiumPackage::findOrFail($id);
            
            $package->update(['is_active' => !$package->is_active]);

            return response()->json([
                'message' => 'Paket durumu başarıyla güncellendi.',
                'is_active' => $package->is_active
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Paket durumu güncellenirken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    /**
     * List packages with pagination (Laravel admin format)
     */
    public function listPackages(Request $request): JsonResponse
    {
        try {
            $query = PremiumPackage::query();

            // Search filter
            if ($request->filled('q')) {
                $search = $request->get('q');
                $query->where('name', 'LIKE', "%{$search}%")
                      ->orWhere('description', 'LIKE', "%{$search}%");
            }

            // Status filter
            if ($request->filled('status')) {
                $query->where('is_active', $request->get('status'));
            }

            // Pagination
            $perPage = $request->get('perPage', 10);
            $page = $request->get('page', 1);
            
            $packages = $query->orderBy('created_at', 'desc')
                             ->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'message' => 'Premium paketleri başarıyla getirildi.',
                'collection' => $packages->items(),
                'pagination' => [
                    'total' => $packages->total(),
                    'per_page' => $packages->perPage(),
                    'current_page' => $packages->currentPage(),
                    'last_page' => $packages->lastPage(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Premium paketleri getirilemedi.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get single package for edit (Laravel admin format)
     */
    public function editPackage(Request $request): JsonResponse
    {
        try {
            $id = $request->get('id');
            $package = PremiumPackage::findOrFail($id);

            return response()->json($package);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Paket bulunamadı.',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Store package (Laravel admin format)
     */
    public function storePackageAdmin(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'duration' => 'required|string|max:255',
                'price' => 'required|numeric|min:0',
                'original_price' => 'nullable|numeric|min:0',
                'description' => 'nullable|string',
                'features' => 'nullable|string',
                'popular' => 'nullable|in:true,false,1,0',
                'color' => 'required|string|in:primary,success,warning,error,info',
                'is_active' => 'nullable|in:true,false,1,0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'Geçersiz veri.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();
            
            // Handle boolean conversions
            $data['popular'] = filter_var($request->get('popular'), FILTER_VALIDATE_BOOLEAN);
            $data['is_active'] = filter_var($request->get('is_active'), FILTER_VALIDATE_BOOLEAN);
            
            // Handle features as JSON string
            if ($request->filled('features')) {
                $features = json_decode($request->get('features'), true);
                $data['features'] = is_array($features) ? $features : [];
            }

            $package = PremiumPackage::create($data);

            return response()->json([
                'message' => 'Premium paket başarıyla oluşturuldu.',
                'data' => $package
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Paket oluşturulurken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update package (Laravel admin format)
     */
    public function updatePackageAdmin(Request $request): JsonResponse
    {
        try {
            $id = $request->get('id');
            $package = PremiumPackage::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'duration' => 'required|string|max:255',
                'price' => 'required|numeric|min:0',
                'original_price' => 'nullable|numeric|min:0',
                'description' => 'nullable|string',
                'features' => 'nullable|string',
                'popular' => 'nullable|in:true,false,1,0',
                'color' => 'required|string|in:primary,success,warning,error,info',
                'is_active' => 'nullable|in:true,false,1,0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'Geçersiz veri.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();
            
            // Handle boolean conversions
            $data['popular'] = filter_var($request->get('popular'), FILTER_VALIDATE_BOOLEAN);
            $data['is_active'] = filter_var($request->get('is_active'), FILTER_VALIDATE_BOOLEAN);
            
            // Handle features as JSON string
            if ($request->filled('features')) {
                $features = json_decode($request->get('features'), true);
                $data['features'] = is_array($features) ? $features : [];
            }

            $package->update($data);

            return response()->json([
                'message' => 'Premium paket başarıyla güncellendi.',
                'data' => $package
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Paket güncellenirken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Destroy package (Laravel admin format)
     */
    public function destroyPackageAdmin(Request $request): JsonResponse
    {
        try {
            $id = $request->get('id');
            $package = PremiumPackage::findOrFail($id);

            // Check if package is used in any subscriptions
            $subscriptionCount = UserPremium::where('package_id', $id)->count();
            
            if ($subscriptionCount > 0) {
                return response()->json([
                    'message' => 'Bu paket kullanımda olduğu için silinemez. Önce pasif hale getirin.'
                ], 422);
            }

            $package->delete();

            return response()->json([
                'message' => 'Premium paket başarıyla silindi.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Paket silinirken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Grant premium to user manually (Admin only)
     */
    public function grantPremiumToUser(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'package_id' => 'required|exists:premium_packages,id',
                'duration_days' => 'required|integer|min:1|max:3650', // Max 10 years
                'reason' => 'nullable|string|max:500'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'Geçersiz veri.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $userId = $request->get('user_id');
            $packageId = $request->get('package_id');
            $durationDays = $request->get('duration_days');
            $reason = $request->get('reason', 'Admin tarafından manuel olarak verildi');

            // Check if user already has active premium
            $existingPremium = UserPremium::where('user_id', $userId)
                ->where('status', 1)
                ->where('end_date', '>', now())
                ->first();

            if ($existingPremium) {
                // Extend existing premium
                $existingPremium->update([
                    'end_date' => $existingPremium->end_date->addDays($durationDays)
                ]);
                
                return response()->json([
                    'message' => "Kullanıcının mevcut premium üyeliği {$durationDays} gün uzatıldı.",
                    'data' => $existingPremium
                ]);
            } else {
                // Create new premium subscription
                $package = PremiumPackage::findOrFail($packageId);
                
                $premium = UserPremium::create([
                    'user_id' => $userId,
                    'package_id' => $packageId,
                    'start_date' => now(),
                    'end_date' => now()->addDays($durationDays),
                    'price' => $package->price,
                    'status' => 1,
                    'payment_method' => 'manual_grant',
                    'notes' => $reason
                ]);

                return response()->json([
                    'message' => "Kullanıcıya {$durationDays} günlük premium üyelik başarıyla verildi.",
                    'data' => $premium
                ]);
            }

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Premium verilirken bir hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search users for premium grant
     */
    public function searchUsers(Request $request): JsonResponse
    {
        try {
            $search = $request->get('search', '');
            \Log::info('🔍 USER SEARCH REQUEST:', [
                'search' => $search,
                'user' => auth()->user()->email ?? 'NO_AUTH',
                'all_params' => $request->all()
            ]);
            
            if (strlen($search) < 2) {
                \Log::info('❌ Search too short, returning empty');
                return response()->json([
                    'data' => []
                ]);
            }

            $users = User::where('name', 'LIKE', "%{$search}%")
                ->orWhere('email', 'LIKE', "%{$search}%")
                ->select('id', 'name', 'email')
                ->limit(10)
                ->get();

            \Log::info('✅ USER SEARCH RESULTS:', [
                'count' => $users->count(),
                'users' => $users->toArray()
            ]);

            return response()->json([
                'data' => $users
            ]);

        } catch (\Exception $e) {
            \Log::error('❌ USER SEARCH ERROR:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'message' => 'Kullanıcı arama sırasında hata oluştu.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get active packages for selection
     */
    public function getActivePackages(): JsonResponse
    {
        try {
            \Log::info('📦 GET ACTIVE PACKAGES REQUEST:', [
                'user' => auth()->user()->email ?? 'NO_AUTH',
            ]);

            $packages = PremiumPackage::where('is_active', true)
                ->select('id', 'name', 'duration', 'price', 'color', 'description')
                ->ordered()
                ->get();

            \Log::info('✅ ACTIVE PACKAGES RESULTS:', [
                'count' => $packages->count(),
                'packages' => $packages->toArray()
            ]);

            return response()->json([
                'data' => $packages
            ]);

        } catch (\Exception $e) {
            \Log::error('❌ GET ACTIVE PACKAGES ERROR:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'message' => 'Aktif paketler getirilemedi.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
