<?php

namespace App\Http\Controllers;

use App\Models\Sliders;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Imagick;

class SliderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Sliders::with(['creator', 'updater']);

        // Search functionality
        if ($request->has('search') && $request->search) {
            $query->where(function($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('subtitle', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
            });
        }

        // Status filter
        if ($request->has('status') && $request->status !== '') {
            $query->where('status', $request->status);
        }

        // Sorting
        if ($request->has('sort_by') && $request->sort_by) {
            $sortOrder = $request->get('sort_order', 'asc');
            $query->orderBy($request->sort_by, $sortOrder);
        } else {
            // Default ordering
            $query->orderBy('order', 'asc')->orderBy('created_at', 'desc');
        }

        // Pagination
        $sliders = $query->paginate($request->get('per_page', 10));

        return response()->json([
            'sliders' => $sliders->items(),
            'total' => $sliders->total(),
            'per_page' => $sliders->perPage(),
            'current_page' => $sliders->currentPage(),
            'last_page' => $sliders->lastPage(),
            'success' => true
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'button_text' => 'nullable|string|max:255',
            'link' => 'nullable|string|max:255',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'nullable|in:0,1,true,false', // FormData boolean fix
            'imagesList.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->all();
        $data['created_by'] = auth()->id();
        $data['updated_by'] = auth()->id();
        $data['status'] = $request->get('is_active', 1); // is_active -> status mapping

        // Handle multiple image uploads (AdsController mantığı)
        if ($request->hasFile('imagesList')) {
            $images = $request->file('imagesList');
            if (is_array($images) && count($images) > 0) {
                // İlk görseli ana görsel olarak kullan
                $image = $images[0];
                $type = $image->extension();
                $filename = time() . '_' . uniqid() . '.' . $type;
                
                // Create directory if not exists
                $galleryPath = storage_path('app/public/uploads/sliders/');
                if (!file_exists($galleryPath)) mkdir($galleryPath, 0777, true);
                
                // Geçici olarak kaydet
                $tempPath = $image->store('temp');
                $fullPath = storage_path('app/' . $tempPath);
                
                // Imagick nesnesi (AdsController gibi)
                $img = new Imagick($fullPath);
                
                // Slider için resize (1200x400)
                $img->resizeImage(1200, 400, Imagick::FILTER_LANCZOS, 1);
                
                // Kaydet
                $img->writeImage($galleryPath . $filename);
                
                $data['image'] = '/storage/uploads/sliders/' . $filename;
                
                // Temp dosyayı sil
                unlink($fullPath);
            }
        }

        // Set order if not provided
        if (!isset($data['order']) || $data['order'] === null) {
            $maxOrder = Sliders::max('order') ?? 0;
            $data['order'] = $maxOrder + 1;
        }

        $slider = Sliders::create($data);

        return response()->json([
            'success' => true,
            'message' => 'Slider başarıyla oluşturuldu.',
            'slider' => $slider->load(['creator', 'updater'])
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $slider = Sliders::with(['creator', 'updater'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'slider' => $slider
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        \Log::info('🟡 SliderController::update - Başlangıç', [
            'id' => $id,
            'request_data' => $request->all(),
            'has_files' => $request->hasFile('imagesList'),
            'auth_id' => auth()->id()
        ]);

        try {
            $slider = Sliders::findOrFail($id);
            \Log::info('🟢 Slider bulundu', ['slider_id' => $slider->id, 'current_image' => $slider->image]);

            $validator = Validator::make($request->all(), [
                'title' => 'required|string|max:255',
                'subtitle' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'button_text' => 'nullable|string|max:255',
                'link' => 'nullable|string|max:255',
                'order' => 'nullable|integer|min:0',
                'is_active' => 'nullable|in:0,1,true,false', // FormData boolean fix
                'imagesList.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048'
            ]);

            if ($validator->fails()) {
                \Log::error('🔴 Validation hatası', ['errors' => $validator->errors()]);
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();
            
            // updated_by alanını kaldır - problem yaratıyor olabilir
            unset($data['updated_by']);
            
            $data['status'] = $request->get('is_active', $slider->status); // is_active -> status mapping
            \Log::info('🟡 Data hazırlandı', ['data_keys' => array_keys($data)]);

            // Handle multiple image uploads (AdsController mantığı)
            if ($request->hasFile('imagesList')) {
                \Log::info('🟡 Görsel yükleme başlıyor');
                $images = $request->file('imagesList');
                \Log::info('🟡 Yüklenen dosya sayısı', ['count' => is_array($images) ? count($images) : 1]);
                
                if (is_array($images) && count($images) > 0) {
                    \Log::info('🟡 Multiple image upload');
                    // Eski görseli sil
                    if ($slider->image && Storage::disk('public')->exists(str_replace('/storage/', '', $slider->image))) {
                        Storage::disk('public')->delete(str_replace('/storage/', '', $slider->image));
                        \Log::info('🟢 Eski görsel silindi');
                    }

                    // İlk görseli ana görsel olarak kullan
                    $image = $images[0];
                    $type = $image->extension();
                    $filename = time() . '_' . uniqid() . '.' . $type;
                    \Log::info('🟡 Yeni dosya adı', ['filename' => $filename]);
                    
                    $galleryPath = storage_path('app/public/uploads/sliders/');
                    if (!file_exists($galleryPath)) {
                        mkdir($galleryPath, 0777, true);
                        \Log::info('🟢 Gallery path oluşturuldu', ['path' => $galleryPath]);
                    }
                    
                    // Geçici olarak kaydet
                    $tempPath = $image->store('temp');
                    $fullPath = storage_path('app/' . $tempPath);
                    \Log::info('🟡 Geçici dosya', ['temp_path' => $fullPath]);
                    
                    // Imagick nesnesi (AdsController gibi)
                    $img = new Imagick($fullPath);
                    \Log::info('🟢 Imagick nesnesi oluşturuldu');
                    
                    // Slider için resize (1200x400)
                    $img->resizeImage(1200, 400, Imagick::FILTER_LANCZOS, 1);
                    \Log::info('🟢 Görsel resize edildi');
                    
                    // Kaydet
                    $img->writeImage($galleryPath . $filename);
                    \Log::info('🟢 Görsel kaydedildi', ['final_path' => $galleryPath . $filename]);
                    
                    $data['image'] = '/storage/uploads/sliders/' . $filename;
                    \Log::info('🟢 Image data atandı', ['image_path' => $data['image']]);
                    
                    // Temp dosyayı sil
                    if (file_exists($fullPath)) {
                        unlink($fullPath);
                        \Log::info('🟢 Temp dosya silindi');
                    }
                } elseif (!is_array($images)) {
                    \Log::info('🟡 Single image upload');
                    // Tek dosya durumu
                    if ($slider->image && Storage::disk('public')->exists(str_replace('/storage/', '', $slider->image))) {
                        Storage::disk('public')->delete(str_replace('/storage/', '', $slider->image));
                        \Log::info('🟢 Eski görsel silindi (single)');
                    }

                    $image = $images;
                    $type = $image->extension();
                    $filename = time() . '_' . uniqid() . '.' . $type;
                    
                    $galleryPath = storage_path('app/public/uploads/sliders/');
                    if (!file_exists($galleryPath)) {
                        mkdir($galleryPath, 0777, true);
                    }
                    
                    // Geçici olarak kaydet
                    $tempPath = $image->store('temp');
                    $fullPath = storage_path('app/' . $tempPath);
                    
                    // Imagick nesnesi
                    $img = new Imagick($fullPath);
                    $img->resizeImage(1200, 400, Imagick::FILTER_LANCZOS, 1);
                    $img->writeImage($galleryPath . $filename);
                    
                    $data['image'] = '/storage/uploads/sliders/' . $filename;
                    \Log::info('🟢 Single image processed', ['image_path' => $data['image']]);
                    
                    // Temp dosyayı sil
                    if (file_exists($fullPath)) {
                        unlink($fullPath);
                    }
                }
            }

            \Log::info('🟡 Slider update işlemi başlıyor', ['final_data' => $data]);
            $slider->update($data);
            
            $updatedSlider = $slider->fresh(['creator', 'updater']);
            \Log::info('🟢 Slider başarıyla güncellendi', [
                'slider_id' => $updatedSlider->id,
                'title' => $updatedSlider->title,
                'image' => $updatedSlider->image,
                'status' => $updatedSlider->status
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Slider başarıyla güncellendi.',
                'slider' => $updatedSlider
            ]);

        } catch (\Exception $e) {
            \Log::error('🔴 SliderController::update - HATA', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Slider güncellenirken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $slider = Sliders::findOrFail($id);

        // Delete image file
        if ($slider->image && Storage::disk('public')->exists(str_replace('/storage/', '', $slider->image))) {
            Storage::disk('public')->delete(str_replace('/storage/', '', $slider->image));
        }

        $slider->delete();

        return response()->json([
            'success' => true,
            'message' => 'Slider başarıyla silindi.'
        ]);
    }

    /**
     * Update slider order
     */
    public function updateOrder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sliders' => 'required|array',
            'sliders.*.id' => 'required|exists:sliders,id',
            'sliders.*.order' => 'required|integer|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        foreach ($request->sliders as $sliderData) {
            Sliders::where('id', $sliderData['id'])->update([
                'order' => $sliderData['order'],
                'updated_by' => auth()->id()
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Slider sıralaması güncellendi.'
        ]);
    }

    /**
     * Toggle slider status
     */
    public function toggleStatus($id)
    {
        $slider = Sliders::findOrFail($id);
        
        $slider->update([
            'status' => !$slider->status,
            'updated_by' => auth()->id()
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Slider durumu güncellendi.',
            'slider' => $slider->fresh(['creator', 'updater'])
        ]);
    }

    /**
     * Get active sliders for frontend
     */
    public function getActiveSliders()
    {
        $sliders = Sliders::active()->ordered()->get();

        return response()->json([
            'success' => true,
            'sliders' => $sliders
        ]);
    }

    /**
     * Gallery destroy function - AdsController.galleryDestroy mantığına birebir uyarlandı
     */
    public function galleryDestroy(Request $request): JsonResponse
    {
        $slider = Sliders::find($request->id);
        if (!$slider){
            return response()->json([
                'status' => 'Error',
                'message' => 'Slider Bulunamadı!'
            ], 400);
        }

        // Delete image file - AdsController mantığı ile aynı
        if ($slider->image && Storage::disk('public')->exists(str_replace('/storage/', '', $slider->image))) {
            Storage::disk('public')->delete(str_replace('/storage/', '', $slider->image));
        }

        // Image dosyası zaten silindi, database'de image field'ını değiştirmeye gerek yok
        // Frontend'de image varlığını dosya sisteminden kontrol edeceğiz

        return response()->json([
            'status' => 'Success',
            'message' => 'Fotoğraf Silindi'
        ], 200);
    }
}
