<?php

namespace App\Http\Controllers;
use App\Http\Resources\User\UserListResource;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $perPage = $request->perPage ?? 10;

        $users = User::paginate($perPage);

        if ($request->query('q')) {
            $q = $request->query('q');
            $users->where(DB::raw('lower(name)'), 'like', '%' . strtolower($q) . '%');
        }

        if ($request->query('status')) {
            $status = $request->query('status') === 'active' ? 1 : 0;
            $users->where('status', $status);
        }

        return response()->json([
            'message' => ('Fetched admin users'),
            'pagination' => $users,
            'collection' => UserListResource::collection($users)
        ]);
    }

    public function store(Request $request): JsonResponse
    {
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);
        $role = Role::where('name', $request->role)->first();
        $user->assignRole($request->role);
        $user->givePermissionTo($role->permissions);


        return response()->json([
            'status' => 'success',
            'message' => 'Kullanıcı Başarılı Şekilde Oluşturuldu.'
        ]);
    }

    public function read(Request $request): JsonResponse
    {
        $user = User::find($request->id);
        $role = $user->getRoleNames()->first();

        return response()->json([
            'user' => $user,
            'role' => $role
        ]);
    }

    public function update(Request $request): JsonResponse
    {
        $user = User::find($request->id);
        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'credit_type' => $request->credit_type,
            'password' => $request->password ? Hash::make($request->password) : $user->password,
        ]);

        $role = Role::where('name', $request->role)->first();
        $user->syncRoles($request->role);
        $user->syncPermissions($role->permissions);


        return response()->json([
            'status' => 'success',
            'message' => 'Kullanıcı Başarılı Şekilde Güncellendi.'
        ]);
    }

    public function suspend(Request $request): JsonResponse
    {
        $user = User::find($request->id);
        $user->status = $request->status === 1 ? 2 : 1;
        $user->save();

        return response()->json([
            'status' => 'success',
            'message' => $request->status == 2 ? 'Kullanıcı Askıdan Kaldırıldı' : 'Kullanıcı Askıya Alındı'
        ]);
    }
}
