import { createApp, h } from 'vue';
import { createBirpcGroup, cachedMap, createBirpc } from 'birpc';

var DevToolsHooks = /* @__PURE__ */ ((DevToolsHooks2) => {
  DevToolsHooks2["APP_INIT"] = "app:init";
  DevToolsHooks2["APP_UNMOUNT"] = "app:unmount";
  DevToolsHooks2["COMPONENT_UPDATED"] = "component:updated";
  DevToolsHooks2["COMPONENT_ADDED"] = "component:added";
  DevToolsHooks2["COMPONENT_REMOVED"] = "component:removed";
  DevToolsHooks2["COMPONENT_EMIT"] = "component:emit";
  DevToolsHooks2["PERFORMANCE_START"] = "perf:start";
  DevToolsHooks2["PERFORMANCE_END"] = "perf:end";
  DevToolsHooks2["ADD_ROUTE"] = "router:add-route";
  DevToolsHooks2["REMOVE_ROUTE"] = "router:remove-route";
  DevToolsHooks2["RENDER_TRACKED"] = "render:tracked";
  DevToolsHooks2["RENDER_TRIGGERED"] = "render:triggered";
  return DevToolsHooks2;
})(DevToolsHooks || {});
function createDevToolsHook() {
  window.__VUE_DEVTOOLS_GLOBAL_HOOK__ ?? (window.__VUE_DEVTOOLS_GLOBAL_HOOK__ = {
    events: /* @__PURE__ */ new Map(),
    on(event, fn) {
      if (!this.events.has(event))
        this.events.set(event, []);
      this.events.get(event).push(fn);
    },
    emit(event, ...payload) {
      if (this.events.has(event))
        this.events.get(event).forEach((fn) => fn(...payload));
    }
  });
  return window.__VUE_DEVTOOLS_GLOBAL_HOOK__;
}
function collectDevToolsHookBuffer() {
  const hookBuffer = [];
  function collect(type, args) {
    hookBuffer.push([type, args]);
  }
  return {
    hookBuffer,
    collect
  };
}

function createDevToolsContainer(App) {
  const hook = createDevToolsHook();
  const CONTAINER_ID = "__vue-devtools-container__";
  const el = document.createElement("div");
  el.setAttribute("id", CONTAINER_ID);
  el.setAttribute("data-v-inspector-ignore", "true");
  document.getElementsByTagName("body")[0].appendChild(el);
  createApp({
    render: () => h(App, { hook }),
    devtools: {
      hide: true
    }
  }).mount(el);
}

function createRPCServer$1(name, ws, functions) {
  const event = `${name}:rpc`;
  const group = createBirpcGroup(
    functions,
    () => cachedMap(
      Array.from(ws?.clients || []),
      (socket) => {
        return {
          on: (fn) => {
            ws.on(event, (data, source) => {
              if (socket === source)
                fn(data);
            });
          },
          post: (data) => {
            socket.send(event, data);
          }
        };
      }
    ),
    {
      timeout: -1
    }
  );
  ws.on("connection", () => {
    group.updateChannels();
  });
  return group.broadcast;
}
function createRPCClient$1(name, hot, functions = {}) {
  const event = `${name}:rpc`;
  return createBirpc(
    functions,
    {
      on: async (fn) => {
        (await hot).on(event, fn);
      },
      post: async (data) => {
        (await hot).send(event, data);
      },
      timeout: -1
    }
  );
}

const PLUGIN_NAME = "vite-plugin-vue-devtools";

function createRPCServer(server, functions) {
  return createRPCServer$1(PLUGIN_NAME, server, functions);
}
function createRPCClient(ctx, functions) {
  return createRPCClient$1(PLUGIN_NAME, ctx, functions);
}

function getComponentTypeName(options) {
  return options.name || options._componentTag || options.__vdevtools_guessedName || options.__name;
}
function saveComponentName(instance, key) {
  return key;
}
function getComponentFileName(options) {
  const file = options.__file;
  if (file) {
    const filename = options.__file?.match(/\/?([^/]+?)(\.[^/.]+)?$/)?.[1];
    return filename ?? file;
  }
}
function getInstanceName(instance) {
  const name = getComponentTypeName(instance.type || {});
  if (name)
    return name;
  if (instance.root === instance)
    return "Root";
  for (const key in instance.parent?.type?.components) {
    if (instance.parent.type.components[key] === instance.type)
      return saveComponentName(instance, key);
  }
  for (const key in instance.appContext?.components) {
    if (instance.appContext.components[key] === instance.type)
      return saveComponentName(instance, key);
  }
  const fileName = getComponentFileName(instance.type || {});
  if (fileName)
    return fileName;
  return "Anonymous Component";
}

export { DevToolsHooks, PLUGIN_NAME, collectDevToolsHookBuffer, createDevToolsContainer, createDevToolsHook, createRPCClient, createRPCServer, getComponentFileName, getComponentTypeName, getInstanceName };
