import { Type } from '../lib/Enum.js';
const NATIVE_TYPES = [
    'String', 'Number', 'Array', 'Boolean', 'Function',
    'Object', 'Null', 'Symbol', 'Undefined', 'BigInt',
];
const NATIVE_TYPES_LIST = NATIVE_TYPES.concat(NATIVE_TYPES.map((type) => type + '[]'));
const NATIVE_TYPES_LOWERCASE = NATIVE_TYPES.map((type) => type.toLowerCase());
const NATIVE_TYPES_ARRAY = NATIVE_TYPES.map((type) => type.toLowerCase() + '[]').concat([Type.any, Type.unknown]);
const NATIVE_TYPES_PREFIX = [
    'Record<',
    'Array<',
    'Map<',
    'Set<',
];
export class Value {
    constructor(type, value, raw) {
        this.type = type || Type.unknown;
        this.value = value;
        this.raw = raw ?? value;
        this.member = false;
        if (this.value instanceof Array) {
            this.rawObject = [];
            this.rawNode = [];
        }
        else if (typeof this.value === 'object' && this.value !== null) {
            this.rawObject = {};
            this.rawNode = {};
        }
    }
    get kind() {
        return Value.parseNativeType(this.type);
    }
    static isNativeType(type) {
        return NATIVE_TYPES.includes(type)
            || NATIVE_TYPES_LOWERCASE.includes(type)
            || NATIVE_TYPES_ARRAY.includes(type)
            || NATIVE_TYPES_PREFIX.some((item) => type.startsWith(item));
    }
    static parseNativeType(type) {
        return NATIVE_TYPES_LIST.includes(type) ? type.toLowerCase() : type;
    }
}
function* undefineGenerator() {
    while (true) {
        yield new Value(Type.unknown, undefined, 'undefined');
    }
}
function* nullGenerator() {
    while (true) {
        yield new Value(Type.unknown, null, 'null');
    }
}
function* objectGenerator() {
    while (true) {
        yield new Value(Type.object, {}, '{}');
    }
}
function* arrayGenerator() {
    while (true) {
        yield new Value(Type.array, [], '[]');
    }
}
export const generateUndefineValue = undefineGenerator();
export const generateNullGenerator = nullGenerator();
export const generateObjectGenerator = objectGenerator();
export const generateArrayGenerator = arrayGenerator();
//# sourceMappingURL=Value.js.map