"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleName = exports.meta = exports.messages = exports["default"] = void 0;
var _styleSearch = _interopRequireDefault(require("style-search"));
var _atRuleParamIndex = _interopRequireDefault(require("../../utils/atRuleParamIndex.js"));
var _declarationValueIndex = _interopRequireDefault(require("../../utils/declarationValueIndex.js"));
var _getDeclarationValue = _interopRequireDefault(require("../../utils/getDeclarationValue.js"));
var _isWhitespace = _interopRequireDefault(require("../../utils/isWhitespace.js"));
var _report = _interopRequireDefault(require("../../utils/report.js"));
var _ruleMessages = _interopRequireDefault(require("../../utils/ruleMessages.js"));
var _setDeclarationValue = _interopRequireDefault(require("../../utils/setDeclarationValue.js"));
var _validateOptions = _interopRequireDefault(require("../../utils/validateOptions.js"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
var ruleName = "function-whitespace-after";
exports.ruleName = ruleName;
var messages = (0, _ruleMessages["default"])(ruleName, {
  expected: "Expected whitespace after \")\"",
  rejected: "Unexpected whitespace after \")\""
});
exports.messages = messages;
var meta = {
  url: "https://github.com/firefoxic/stylelint-codeguide/blob/main/lib/rules/function-whitespace-after/README.md",
  fixable: true
};
exports.meta = meta;
var ACCEPTABLE_AFTER_CLOSING_PAREN = new Set([")", ",", "}", ":", "/", undefined]);

/** @type {import('stylelint').Rule} */
var rule = function rule(primary, _secondaryOptions, context) {
  return function (root, result) {
    var validOptions = (0, _validateOptions["default"])(result, ruleName, {
      actual: primary,
      possible: ["always", "never"]
    });
    if (!validOptions) {
      return;
    }

    /**
    	 * @param {import('postcss').Node} node
    	 * @param {string} value
    	 * @param {number} nodeIndex
    	 * @param {((index: number) => void) | undefined} fix
    	 */
    function check(node, value, nodeIndex, fix) {
      (0, _styleSearch["default"])({
        source: value,
        target: ")",
        functionArguments: "only"
      }, function (match) {
        checkClosingParen(value, match.startIndex + 1, node, nodeIndex, fix);
      });
    }

    /**
    	 * @param {string} source
    	 * @param {number} index
    	 * @param {import('postcss').Node} node
    	 * @param {number} nodeIndex
    	 * @param {((index: number) => void) | undefined} fix
    	 */
    function checkClosingParen(source, index, node, nodeIndex, fix) {
      var nextChar = source.charAt(index);
      if (!nextChar) {
        return;
      }
      if (primary === "always") {
        // Allow for the next character to be a single empty space,
        // another closing parenthesis, a comma, or the end of the value
        if (nextChar === " ") {
          return;
        }
        if (nextChar === "\n") {
          return;
        }
        if (source.slice(index, index + 2) === "\r\n") {
          return;
        }
        if (ACCEPTABLE_AFTER_CLOSING_PAREN.has(nextChar)) {
          return;
        }
        if (fix) {
          fix(index);
          return;
        }
        (0, _report["default"])({
          message: messages.expected,
          node: node,
          index: nodeIndex + index,
          result: result,
          ruleName: ruleName
        });
      } else if (primary === "never" && (0, _isWhitespace["default"])(nextChar)) {
        if (fix) {
          fix(index);
          return;
        }
        (0, _report["default"])({
          message: messages.rejected,
          node: node,
          index: nodeIndex + index,
          result: result,
          ruleName: ruleName
        });
      }
    }

    /**
    	 * @param {string} value
    	 */
    function createFixer(value) {
      var fixed = "";
      var lastIndex = 0;
      /** @type {(index: number) => void} */
      var applyFix;
      if (primary === "always") {
        applyFix = function applyFix(index) {
          fixed += "".concat(value.slice(lastIndex, index), " ");
          lastIndex = index;
        };
      } else if (primary === "never") {
        applyFix = function applyFix(index) {
          var whitespaceEndIndex = index + 1;
          while (whitespaceEndIndex < value.length && (0, _isWhitespace["default"])(value.charAt(whitespaceEndIndex))) {
            whitespaceEndIndex++;
          }
          fixed += value.slice(lastIndex, index);
          lastIndex = whitespaceEndIndex;
        };
      } else {
        throw new Error("Unexpected option: \"".concat(primary, "\""));
      }
      return {
        applyFix: applyFix,
        get hasFixed() {
          return Boolean(lastIndex);
        },
        get fixed() {
          return fixed + value.slice(lastIndex);
        }
      };
    }
    root.walkAtRules(/^import$/i, function (atRule) {
      var param = atRule.raws.params && atRule.raws.params.raw || atRule.params;
      var fixer = context.fix && createFixer(param);
      check(atRule, param, (0, _atRuleParamIndex["default"])(atRule), fixer ? fixer.applyFix : undefined);
      if (fixer && fixer.hasFixed) {
        if (atRule.raws.params) {
          atRule.raws.params.raw = fixer.fixed;
        } else {
          atRule.params = fixer.fixed;
        }
      }
    });
    root.walkDecls(function (decl) {
      var value = (0, _getDeclarationValue["default"])(decl);
      var fixer = context.fix && createFixer(value);
      check(decl, value, (0, _declarationValueIndex["default"])(decl), fixer ? fixer.applyFix : undefined);
      if (fixer && fixer.hasFixed) {
        (0, _setDeclarationValue["default"])(decl, fixer.fixed);
      }
    });
  };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
var _default = rule;
exports["default"] = _default;