"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleName = exports.meta = exports.messages = exports["default"] = void 0;
var _styleSearch = _interopRequireDefault(require("style-search"));
var _optionsMatches = _interopRequireDefault(require("../../utils/optionsMatches.js"));
var _report = _interopRequireDefault(require("../../utils/report.js"));
var _ruleMessages = _interopRequireDefault(require("../../utils/ruleMessages.js"));
var _validateOptions = _interopRequireDefault(require("../../utils/validateOptions.js"));
var _validateTypes = require("../../utils/validateTypes.js");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
var ruleName = "max-empty-lines";
exports.ruleName = ruleName;
var messages = (0, _ruleMessages["default"])(ruleName, {
  expected: function expected(max) {
    return "Expected no more than ".concat(max, " empty ").concat(max === 1 ? "line" : "lines");
  }
});
exports.messages = messages;
var meta = {
  url: "https://github.com/firefoxic/stylelint-codeguide/blob/main/lib/rules/max-empty-lines/README.md",
  fixable: true
};

/** @type {import('stylelint').Rule} */
exports.meta = meta;
var rule = function rule(primary, secondaryOptions, context) {
  var emptyLines = 0;
  var lastIndex = -1;
  return function (root, result) {
    var validOptions = (0, _validateOptions["default"])(result, ruleName, {
      actual: primary,
      possible: _validateTypes.isNumber
    }, {
      actual: secondaryOptions,
      possible: {
        ignore: ["comments"]
      },
      optional: true
    });
    if (!validOptions) {
      return;
    }
    var ignoreComments = (0, _optionsMatches["default"])(secondaryOptions, "ignore", "comments");
    var getChars = replaceEmptyLines.bind(null, primary);

    /**
     * 1. walk nodes & replace enterchar
     * 2. deal with special case.
     */
    if (context.fix) {
      root.walk(function (node) {
        if (node.type === "comment" && !ignoreComments) {
          node.raws.left = getChars(node.raws.left);
          node.raws.right = getChars(node.raws.right);
        }
        if (node.raws.before) {
          node.raws.before = getChars(node.raws.before);
        }
      });

      // first node
      var firstNodeRawsBefore = root.first && root.first.raws.before;
      // root raws
      var rootRawsAfter = root.raws.after;

      // not document node
      // @ts-expect-error -- TS2339: Property 'document' does not exist on type 'Root'.
      if ((root.document && root.document.constructor.name) !== "Document") {
        if (firstNodeRawsBefore) {
          root.first.raws.before = getChars(firstNodeRawsBefore, true);
        }
        if (rootRawsAfter) {
          // when max setted 0, should be treated as 1 in this situation.
          root.raws.after = replaceEmptyLines(primary === 0 ? 1 : primary, rootRawsAfter, true);
        }
      } else if (rootRawsAfter) {
        // `css in js` or `html`
        root.raws.after = replaceEmptyLines(primary === 0 ? 1 : primary, rootRawsAfter);
      }
      return;
    }
    emptyLines = 0;
    lastIndex = -1;
    var rootString = root.toString();
    (0, _styleSearch["default"])({
      source: rootString,
      target: /\r\n/.test(rootString) ? "\r\n" : "\n",
      comments: ignoreComments ? "skip" : "check"
    }, function (match) {
      checkMatch(rootString, match.startIndex, match.endIndex, root);
    });

    /**
     * @param {string} source
     * @param {number} matchStartIndex
     * @param {number} matchEndIndex
     * @param {import('postcss').Root} node
     */
    function checkMatch(source, matchStartIndex, matchEndIndex, node) {
      var eof = matchEndIndex === source.length;
      var problem = false;

      // Additional check for beginning of file
      if (!matchStartIndex || lastIndex === matchStartIndex) {
        emptyLines++;
      } else {
        emptyLines = 0;
      }
      lastIndex = matchEndIndex;
      if (emptyLines > primary) {
        problem = true;
      }
      if (!eof && !problem) {
        return;
      }
      if (problem) {
        (0, _report["default"])({
          message: messages.expected(primary),
          node: node,
          index: matchStartIndex,
          result: result,
          ruleName: ruleName
        });
      }

      // Additional check for end of file
      if (eof && primary) {
        emptyLines++;
        if (emptyLines > primary && isEofNode(result.root, node)) {
          (0, _report["default"])({
            message: messages.expected(primary),
            node: node,
            index: matchEndIndex,
            result: result,
            ruleName: ruleName
          });
        }
      }
    }

    /**
     * @param {number} maxLines
     * @param {unknown} str
     * @param {boolean?} isSpecialCase
     */
    function replaceEmptyLines(maxLines, str) {
      var isSpecialCase = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
      var repeatTimes = isSpecialCase ? maxLines : maxLines + 1;
      if (repeatTimes === 0 || _typeof(str) !== "string") {
        return "";
      }
      var emptyLFLines = "\n".repeat(repeatTimes);
      var emptyCRLFLines = "\r\n".repeat(repeatTimes);
      return /(?:\r\n)+/.test(str) ? str.replace(/(\r\n)+/g, function ($1) {
        if ($1.length / 2 > repeatTimes) {
          return emptyCRLFLines;
        }
        return $1;
      }) : str.replace(/(\n)+/g, function ($1) {
        if ($1.length > repeatTimes) {
          return emptyLFLines;
        }
        return $1;
      });
    }
  };
};

/**
 * Checks whether the given node is the last node of file.
 * @param {import('stylelint').PostcssResult['root']} document - the document node with `postcss-html` and `postcss-jsx`.
 * @param {import('postcss').Root} root - the root node of css
 */
function isEofNode(document, root) {
  if (!document || document.constructor.name !== "Document" || !("type" in document)) {
    return true;
  }

  // In the `postcss-html` and `postcss-jsx` syntax, checks that there is text after the given node.
  var after;
  if (root === document.last) {
    after = document.raws && document.raws.codeAfter;
  } else {
    // @ts-expect-error -- TS2345: Argument of type 'Root' is not assignable to parameter of type 'number | ChildNode'.
    var rootIndex = document.index(root);
    var nextNode = document.nodes[rootIndex + 1];
    after = nextNode && nextNode.raws && nextNode.raws.codeBefore;
  }
  return !String(after).trim();
}
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
var _default = rule;
exports["default"] = _default;