"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleName = exports.meta = exports.messages = exports["default"] = void 0;
var _styleSearch = _interopRequireDefault(require("style-search"));
var _isOnlyWhitespace = _interopRequireDefault(require("../../utils/isOnlyWhitespace.js"));
var _isStandardSyntaxComment = _interopRequireDefault(require("../../utils/isStandardSyntaxComment.js"));
var _optionsMatches = _interopRequireDefault(require("../../utils/optionsMatches.js"));
var _report = _interopRequireDefault(require("../../utils/report.js"));
var _ruleMessages = _interopRequireDefault(require("../../utils/ruleMessages.js"));
var _typeGuards = require("../../utils/typeGuards.js");
var _validateOptions = _interopRequireDefault(require("../../utils/validateOptions.js"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
var ruleName = "no-eol-whitespace";
exports.ruleName = ruleName;
var messages = (0, _ruleMessages["default"])(ruleName, {
  rejected: "Unexpected whitespace at end of line"
});
exports.messages = messages;
var meta = {
  url: "https://github.com/firefoxic/stylelint-codeguide/blob/main/lib/rules/no-eol-whitespace/README.md",
  fixable: true
};
exports.meta = meta;
var whitespacesToReject = new Set([" ", "\t"]);

/**
 * @param {string} str
 * @returns {string}
 */
function fixString(str) {
  return str.replace(/[ \t]+$/, "");
}

/**
 * @param {number} lastEOLIndex
 * @param {string} string
 * @param {{ ignoreEmptyLines: boolean, isRootFirst: boolean }} options
 * @returns {number}
 */
function findErrorStartIndex(lastEOLIndex, string, _ref) {
  var ignoreEmptyLines = _ref.ignoreEmptyLines,
    isRootFirst = _ref.isRootFirst;
  var eolWhitespaceIndex = lastEOLIndex - 1;

  // If the character before newline is not whitespace, ignore
  if (!whitespacesToReject.has(string.charAt(eolWhitespaceIndex))) {
    return -1;
  }
  if (ignoreEmptyLines) {
    // If there is only whitespace between the previous newline and
    // this newline, ignore
    var beforeNewlineIndex = string.lastIndexOf("\n", eolWhitespaceIndex);
    if (beforeNewlineIndex >= 0 || isRootFirst) {
      var line = string.substring(beforeNewlineIndex, eolWhitespaceIndex);
      if ((0, _isOnlyWhitespace["default"])(line)) {
        return -1;
      }
    }
  }
  return eolWhitespaceIndex;
}

/** @type {import('stylelint').Rule} */
var rule = function rule(primary, secondaryOptions, context) {
  return function (root, result) {
    var validOptions = (0, _validateOptions["default"])(result, ruleName, {
      actual: primary
    }, {
      optional: true,
      actual: secondaryOptions,
      possible: {
        ignore: ["empty-lines"]
      }
    });
    if (!validOptions) {
      return;
    }
    var ignoreEmptyLines = (0, _optionsMatches["default"])(secondaryOptions, "ignore", "empty-lines");
    if (context.fix) {
      fix(root);
    }
    var rootString = context.fix ? root.toString() : root.source && root.source.input.css || "";

    /**
    	 * @param {number} index
    	 */
    var reportFromIndex = function reportFromIndex(index) {
      (0, _report["default"])({
        message: messages.rejected,
        node: root,
        index: index,
        result: result,
        ruleName: ruleName
      });
    };
    eachEolWhitespace(rootString, reportFromIndex, true);
    var errorIndex = findErrorStartIndex(rootString.length, rootString, {
      ignoreEmptyLines: ignoreEmptyLines,
      isRootFirst: true
    });
    if (errorIndex > -1) {
      reportFromIndex(errorIndex);
    }

    /**
    	 * Iterate each whitespace at the end of each line of the given string.
    	 * @param {string} string - the source code string
    	 * @param {(index: number) => void} callback - callback the whitespace index at the end of each line.
    	 * @param {boolean} isRootFirst - set `true` if the given string is the first token of the root.
    	 * @returns {void}
    	 */
    function eachEolWhitespace(string, callback, isRootFirst) {
      (0, _styleSearch["default"])({
        source: string,
        target: ["\n", "\r"],
        comments: "check"
      }, function (match) {
        var index = findErrorStartIndex(match.startIndex, string, {
          ignoreEmptyLines: ignoreEmptyLines,
          isRootFirst: isRootFirst
        });
        if (index > -1) {
          callback(index);
        }
      });
    }

    /**
    	 * @param {import('postcss').Root} rootNode
    	 */
    function fix(rootNode) {
      var isRootFirst = true;
      rootNode.walk(function (node) {
        fixText(node.raws.before, function (fixed) {
          node.raws.before = fixed;
        }, isRootFirst);
        isRootFirst = false;
        if ((0, _typeGuards.isAtRule)(node)) {
          fixText(node.raws.afterName, function (fixed) {
            node.raws.afterName = fixed;
          });
          var rawsParams = node.raws.params;
          if (rawsParams) {
            fixText(rawsParams.raw, function (fixed) {
              rawsParams.raw = fixed;
            });
          } else {
            fixText(node.params, function (fixed) {
              node.params = fixed;
            });
          }
        }
        if ((0, _typeGuards.isRule)(node)) {
          var rawsSelector = node.raws.selector;
          if (rawsSelector) {
            fixText(rawsSelector.raw, function (fixed) {
              rawsSelector.raw = fixed;
            });
          } else {
            fixText(node.selector, function (fixed) {
              node.selector = fixed;
            });
          }
        }
        if ((0, _typeGuards.isAtRule)(node) || (0, _typeGuards.isRule)(node) || (0, _typeGuards.isDeclaration)(node)) {
          fixText(node.raws.between, function (fixed) {
            node.raws.between = fixed;
          });
        }
        if ((0, _typeGuards.isDeclaration)(node)) {
          var rawsValue = node.raws.value;
          if (rawsValue) {
            fixText(rawsValue.raw, function (fixed) {
              rawsValue.raw = fixed;
            });
          } else {
            fixText(node.value, function (fixed) {
              node.value = fixed;
            });
          }
        }
        if ((0, _typeGuards.isComment)(node)) {
          fixText(node.raws.left, function (fixed) {
            node.raws.left = fixed;
          });
          if (!(0, _isStandardSyntaxComment["default"])(node)) {
            node.raws.right = node.raws.right && fixString(node.raws.right);
          } else {
            fixText(node.raws.right, function (fixed) {
              node.raws.right = fixed;
            });
          }
          fixText(node.text, function (fixed) {
            node.text = fixed;
          });
        }
        if ((0, _typeGuards.isAtRule)(node) || (0, _typeGuards.isRule)(node)) {
          fixText(node.raws.after, function (fixed) {
            node.raws.after = fixed;
          });
        }
      });
      fixText(rootNode.raws.after, function (fixed) {
        rootNode.raws.after = fixed;
      }, isRootFirst);
      if (_typeof(rootNode.raws.after) === "string") {
        var lastEOL = Math.max(rootNode.raws.after.lastIndexOf("\n"), rootNode.raws.after.lastIndexOf("\r"));
        if (lastEOL !== rootNode.raws.after.length - 1) {
          rootNode.raws.after = rootNode.raws.after.slice(0, lastEOL + 1) + fixString(rootNode.raws.after.slice(lastEOL + 1));
        }
      }
    }

    /**
    	 * @param {string | undefined} value
    	 * @param {(text: string) => void} fixFn
    	 * @param {boolean} isRootFirst
    	 */
    function fixText(value, fixFn) {
      var isRootFirst = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
      if (!value) {
        return;
      }
      var fixed = "";
      var lastIndex = 0;
      eachEolWhitespace(value, function (index) {
        var newlineIndex = index + 1;
        fixed += fixString(value.slice(lastIndex, newlineIndex));
        lastIndex = newlineIndex;
      }, isRootFirst);
      if (lastIndex) {
        fixed += value.slice(lastIndex);
        fixFn(fixed);
      }
    }
  };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
var _default = rule;
exports["default"] = _default;