"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = rangeContextNodeParser;
var _postcssValueParser = _interopRequireDefault(require("postcss-value-parser"));
var _validateTypes = require("../utils/validateTypes.js");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
var rangeOperators = new Set([">=", "<=", ">", "<", "="]);

/**
 * @param {string} name
 * @returns {boolean}
 */
function isRangeContextName(name) {
  // When the node is like "(width > 10em)" or "(10em < width)"
  // Regex is needed because the name can either be in the first or second position
  return /^(?!--)\D/.test(name) || /^--./.test(name);
}

/**
 * @typedef {{ value: string, sourceIndex: number }} RangeContextNode
 *
 * @param {import('postcss-media-query-parser').Node} node
 * @returns {{ name: RangeContextNode, values: RangeContextNode[] }}
 */
function rangeContextNodeParser(node) {
  /** @type {import('postcss-value-parser').WordNode | undefined} */
  var nameNode;

  /** @type {import('postcss-value-parser').WordNode[]} */
  var valueNodes = [];
  (0, _postcssValueParser["default"])(node.value).walk(function (valueNode) {
    if (valueNode.type !== "word") {
      return;
    }
    if (rangeOperators.has(valueNode.value)) {
      return;
    }
    if (nameNode === null && isRangeContextName(valueNode.value)) {
      nameNode = valueNode;
      return;
    }
    valueNodes.push(valueNode);
  });
  (0, _validateTypes.assert)(nameNode);
  return {
    name: {
      value: nameNode.value,
      sourceIndex: node.sourceIndex + nameNode.sourceIndex
    },
    values: valueNodes.map(function (valueNode) {
      return {
        value: valueNode.value,
        sourceIndex: node.sourceIndex + valueNode.sourceIndex
      };
    })
  };
}