"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleName = exports.meta = exports.messages = exports["default"] = void 0;
var _isStandardSyntaxRule = _interopRequireDefault(require("../../utils/isStandardSyntaxRule.js"));
var _parseSelector = _interopRequireDefault(require("../../utils/parseSelector.js"));
var _report = _interopRequireDefault(require("../../utils/report.js"));
var _ruleMessages = _interopRequireDefault(require("../../utils/ruleMessages.js"));
var _validateOptions = _interopRequireDefault(require("../../utils/validateOptions.js"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
var ruleName = "selector-pseudo-class-parentheses-space-inside";
exports.ruleName = ruleName;
var messages = (0, _ruleMessages["default"])(ruleName, {
  expectedOpening: "Expected single space after \"(\"",
  rejectedOpening: "Unexpected whitespace after \"(\"",
  expectedClosing: "Expected single space before \")\"",
  rejectedClosing: "Unexpected whitespace before \")\""
});
exports.messages = messages;
var meta = {
  url: "https://github.com/firefoxic/stylelint-codeguide/blob/main/lib/rules/selector-pseudo-class-parentheses-space-inside/README.md",
  fixable: true
};

/** @type {import('stylelint').Rule} */
exports.meta = meta;
var rule = function rule(primary, _secondaryOptions, context) {
  return function (root, result) {
    var validOptions = (0, _validateOptions["default"])(result, ruleName, {
      actual: primary,
      possible: ["always", "never"]
    });
    if (!validOptions) {
      return;
    }
    root.walkRules(function (ruleNode) {
      if (!(0, _isStandardSyntaxRule["default"])(ruleNode)) {
        return;
      }
      if (!ruleNode.selector.includes("(")) {
        return;
      }
      var hasFixed = false;
      var selector = ruleNode.raws.selector ? ruleNode.raws.selector.raw : ruleNode.selector;
      var fixedSelector = (0, _parseSelector["default"])(selector, result, ruleNode, function (selectorTree) {
        selectorTree.walkPseudos(function (pseudoNode) {
          if (!pseudoNode.length) {
            return;
          }
          var paramString = pseudoNode.map(function (node) {
            return String(node);
          }).join(",");
          var nextCharIsSpace = paramString.startsWith(" ");
          var openIndex = pseudoNode.sourceIndex + pseudoNode.value.length + 1;
          if (nextCharIsSpace && primary === "never") {
            if (context.fix) {
              hasFixed = true;
              setFirstNodeSpaceBefore(pseudoNode, "");
            } else {
              complain(messages.rejectedOpening, openIndex);
            }
          }
          if (!nextCharIsSpace && primary === "always") {
            if (context.fix) {
              hasFixed = true;
              setFirstNodeSpaceBefore(pseudoNode, " ");
            } else {
              complain(messages.expectedOpening, openIndex);
            }
          }
          var prevCharIsSpace = paramString.endsWith(" ");
          var closeIndex = openIndex + paramString.length - 1;
          if (prevCharIsSpace && primary === "never") {
            if (context.fix) {
              hasFixed = true;
              setLastNodeSpaceAfter(pseudoNode, "");
            } else {
              complain(messages.rejectedClosing, closeIndex);
            }
          }
          if (!prevCharIsSpace && primary === "always") {
            if (context.fix) {
              hasFixed = true;
              setLastNodeSpaceAfter(pseudoNode, " ");
            } else {
              complain(messages.expectedClosing, closeIndex);
            }
          }
        });
      });
      if (hasFixed && fixedSelector) {
        if (!ruleNode.raws.selector) {
          ruleNode.selector = fixedSelector;
        } else {
          ruleNode.raws.selector.raw = fixedSelector;
        }
      }

      /**
      	 * @param {string} message
      	 * @param {number} index
      	 */
      function complain(message, index) {
        (0, _report["default"])({
          message: message,
          index: index,
          result: result,
          ruleName: ruleName,
          node: ruleNode
        });
      }
    });
  };
};

/**
 * @param {import('postcss-selector-parser').Container} node
 * @param {string} value
 * @returns {void}
 */
function setFirstNodeSpaceBefore(node, value) {
  var target = node.first;
  if (target.type === "selector") {
    setFirstNodeSpaceBefore(target, value);
  } else {
    target.spaces.before = value;
  }
}

/**
 * @param {import('postcss-selector-parser').Container} node
 * @param {string} value
 * @returns {void}
 */
function setLastNodeSpaceAfter(node, value) {
  var target = node.last;
  if (target.type === "selector") {
    setLastNodeSpaceAfter(target, value);
  } else {
    target.spaces.after = value;
  }
}
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
var _default = rule;
exports["default"] = _default;