"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = selectorCombinatorSpaceChecker;
var _isStandardSyntaxCombinator = _interopRequireDefault(require("../utils/isStandardSyntaxCombinator.js"));
var _isStandardSyntaxRule = _interopRequireDefault(require("../utils/isStandardSyntaxRule.js"));
var _parseSelector = _interopRequireDefault(require("../utils/parseSelector.js"));
var _report = _interopRequireDefault(require("../utils/report.js"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
/**
 * @typedef {(args: { source: string, index: number, errTarget: string, err: (message: string) => void }) => void} LocationChecker
 *
 * @param {{
 *   root: import('postcss').Root,
 *   result: import('stylelint').PostcssResult,
 *   locationChecker: LocationChecker,
 *   locationType: 'before' | 'after',
 *   checkedRuleName: string,
 *   fix: ((combinator: import('postcss-selector-parser').Combinator) => boolean) | null,
 * }} opts
 * @returns {void}
 */
function selectorCombinatorSpaceChecker(opts) {
  var hasFixed;
  opts.root.walkRules(function (rule) {
    if (!(0, _isStandardSyntaxRule["default"])(rule)) {
      return;
    }
    hasFixed = false;
    var selector = rule.raws.selector ? rule.raws.selector.raw : rule.selector;
    var fixedSelector = (0, _parseSelector["default"])(selector, opts.result, rule, function (selectorTree) {
      selectorTree.walkCombinators(function (node) {
        // Ignore non-standard combinators
        if (!(0, _isStandardSyntaxCombinator["default"])(node)) {
          return;
        }

        // Ignore spaced descendant combinator
        if (/\s/.test(node.value)) {
          return;
        }

        // Check the exist of node in prev of the combinator.
        // in case some that aren't the first begin with combinators (nesting syntax)
        if (opts.locationType === "before" && !node.prev()) {
          return;
        }
        var parentParentNode = node.parent && node.parent.parent;

        // Ignore pseudo-classes selector like `.foo:nth-child(2n + 1) {}`
        if (parentParentNode && parentParentNode.type === "pseudo") {
          return;
        }
        var sourceIndex = node.sourceIndex;
        var index = node.value.length > 1 && opts.locationType === "before" ? sourceIndex : sourceIndex + node.value.length - 1;
        check(selector, node, index, rule, sourceIndex);
      });
    });
    if (hasFixed && fixedSelector) {
      if (!rule.raws.selector) {
        rule.selector = fixedSelector;
      } else {
        rule.raws.selector.raw = fixedSelector;
      }
    }
  });

  /**
   * @param {string} source
   * @param {import('postcss-selector-parser').Combinator} combinator
   * @param {number} index
   * @param {import('postcss').Node} node
   * @param {number} sourceIndex
   */
  function check(source, combinator, index, node, sourceIndex) {
    opts.locationChecker({
      source: source,
      index: index,
      errTarget: combinator.value,
      err: function err(message) {
        if (opts.fix && opts.fix(combinator)) {
          hasFixed = true;
          return;
        }
        (0, _report["default"])({
          message: message,
          node: node,
          index: sourceIndex,
          result: opts.result,
          ruleName: opts.checkedRuleName
        });
      }
    });
  }
}