"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleName = exports.meta = exports.messages = exports["default"] = void 0;
var _postcssValueParser = _interopRequireDefault(require("postcss-value-parser"));
var _atRuleParamIndex = _interopRequireDefault(require("../../utils/atRuleParamIndex.js"));
var _declarationValueIndex = _interopRequireDefault(require("../../utils/declarationValueIndex.js"));
var _isStandardSyntaxRule = _interopRequireDefault(require("../../utils/isStandardSyntaxRule.js"));
var _parseSelector = _interopRequireDefault(require("../../utils/parseSelector.js"));
var _report = _interopRequireDefault(require("../../utils/report.js"));
var _ruleMessages = _interopRequireDefault(require("../../utils/ruleMessages.js"));
var _validateOptions = _interopRequireDefault(require("../../utils/validateOptions.js"));
var _validateTypes = require("../../utils/validateTypes.js");
var _typeGuards = require("../../utils/typeGuards.js");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
var ruleName = "string-quotes";
exports.ruleName = ruleName;
var messages = (0, _ruleMessages["default"])(ruleName, {
  expected: function expected(q) {
    return "Expected ".concat(q, " quotes");
  }
});
exports.messages = messages;
var meta = {
  url: "https://github.com/firefoxic/stylelint-codeguide/blob/main/lib/rules/string-quotes/README.md",
  fixable: true
};
exports.meta = meta;
var singleQuote = "'";
var doubleQuote = "\"";

/** @type {import('stylelint').Rule} */
var rule = function rule(primary, secondaryOptions, context) {
  var correctQuote = primary === "single" ? singleQuote : doubleQuote;
  var erroneousQuote = primary === "single" ? doubleQuote : singleQuote;
  return function (root, result) {
    var validOptions = (0, _validateOptions["default"])(result, ruleName, {
      actual: primary,
      possible: ["single", "double"]
    }, {
      actual: secondaryOptions,
      possible: {
        avoidEscape: [_validateTypes.isBoolean]
      },
      optional: true
    });
    if (!validOptions) {
      return;
    }
    var avoidEscape = secondaryOptions && secondaryOptions.avoidEscape !== undefined ? secondaryOptions.avoidEscape : true;
    root.walk(function (node) {
      switch (node.type) {
        case "atrule":
          checkDeclOrAtRule(node, node.params, _atRuleParamIndex["default"]);
          break;
        case "decl":
          checkDeclOrAtRule(node, node.value, _declarationValueIndex["default"]);
          break;
        case "rule":
          checkRule(node);
          break;
      }
    });

    /**
     * @param {import('postcss').Rule} ruleNode
     * @returns {void}
     */
    function checkRule(ruleNode) {
      if (!(0, _isStandardSyntaxRule["default"])(ruleNode)) {
        return;
      }
      if (!ruleNode.selector.includes("[") || !ruleNode.selector.includes("=")) {
        return;
      }

      /** @type {number[]} */
      var fixPositions = [];
      (0, _parseSelector["default"])(ruleNode.selector, result, ruleNode, function (selectorTree) {
        var selectorFixed = false;
        selectorTree.walkAttributes(function (attributeNode) {
          if (!attributeNode.quoted) {
            return;
          }
          if (attributeNode.quoteMark === correctQuote && avoidEscape) {
            (0, _validateTypes.assertString)(attributeNode.value);
            var needsCorrectEscape = attributeNode.value.includes(correctQuote);
            var needsOtherEscape = attributeNode.value.includes(erroneousQuote);
            if (needsOtherEscape) {
              return;
            }
            if (needsCorrectEscape) {
              if (context.fix) {
                selectorFixed = true;
                attributeNode.quoteMark = erroneousQuote;
              } else {
                (0, _report["default"])({
                  message: messages.expected(primary === "single" ? "double" : primary),
                  node: ruleNode,
                  index: attributeNode.sourceIndex + attributeNode.offsetOf("value"),
                  result: result,
                  ruleName: ruleName
                });
              }
            }
          }
          if (attributeNode.quoteMark === erroneousQuote) {
            if (avoidEscape) {
              (0, _validateTypes.assertString)(attributeNode.value);
              var _needsCorrectEscape = attributeNode.value.includes(correctQuote);
              var _needsOtherEscape = attributeNode.value.includes(erroneousQuote);
              if (_needsOtherEscape) {
                if (context.fix) {
                  selectorFixed = true;
                  attributeNode.quoteMark = correctQuote;
                } else {
                  (0, _report["default"])({
                    message: messages.expected(primary),
                    node: ruleNode,
                    index: attributeNode.sourceIndex + attributeNode.offsetOf("value"),
                    result: result,
                    ruleName: ruleName
                  });
                }
                return;
              }
              if (_needsCorrectEscape) {
                return;
              }
            }
            if (context.fix) {
              selectorFixed = true;
              attributeNode.quoteMark = correctQuote;
            } else {
              (0, _report["default"])({
                message: messages.expected(primary),
                node: ruleNode,
                index: attributeNode.sourceIndex + attributeNode.offsetOf("value"),
                result: result,
                ruleName: ruleName
              });
            }
          }
        });
        if (selectorFixed) {
          ruleNode.selector = selectorTree.toString();
        }
      });
      for (var _i = 0, _fixPositions = fixPositions; _i < _fixPositions.length; _i++) {
        var fixIndex = _fixPositions[_i];
        ruleNode.selector = replaceQuote(ruleNode.selector, fixIndex, correctQuote);
      }
    }

    /**
     * @template {import('postcss').AtRule | import('postcss').Declaration} T
     * @param {T} node
     * @param {string} value
     * @param {(node: T) => number} getIndex
     * @returns {void}
     */
    function checkDeclOrAtRule(node, value, getIndex) {
      /** @type {number[]} */
      var fixPositions = [];

      // Get out quickly if there are no erroneous quotes
      if (!value.includes(erroneousQuote)) {
        return;
      }
      if ((0, _typeGuards.isAtRule)(node) && node.name === "charset") {
        // allow @charset rules to have double quotes, in spite of the configuration
        // TODO: @charset should always use double-quotes, see https://github.com/stylelint/stylelint/issues/2788
        return;
      }
      (0, _postcssValueParser["default"])(value).walk(function (valueNode) {
        if (valueNode.type === "string" && valueNode.quote === erroneousQuote) {
          var needsEscape = valueNode.value.includes(correctQuote);
          if (avoidEscape && needsEscape) {
            // don't consider this an error
            return;
          }
          var openIndex = valueNode.sourceIndex;

          // we currently don't fix escapes
          if (context.fix && !needsEscape) {
            var closeIndex = openIndex + valueNode.value.length + erroneousQuote.length;
            fixPositions.push(openIndex, closeIndex);
          } else {
            (0, _report["default"])({
              message: messages.expected(primary),
              node: node,
              index: getIndex(node) + openIndex,
              result: result,
              ruleName: ruleName
            });
          }
        }
      });
      for (var _i2 = 0, _fixPositions2 = fixPositions; _i2 < _fixPositions2.length; _i2++) {
        var fixIndex = _fixPositions2[_i2];
        if ((0, _typeGuards.isAtRule)(node)) {
          node.params = replaceQuote(node.params, fixIndex, correctQuote);
        } else {
          node.value = replaceQuote(node.value, fixIndex, correctQuote);
        }
      }
    }
  };
};

/**
 * @param {string} string
 * @param {number} index
 * @param {string} replace
 * @returns {string}
 */
function replaceQuote(string, index, replace) {
  return string.substring(0, index) + replace + string.substring(index + replace.length);
}
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
var _default = rule;
exports["default"] = _default;