"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = whitespaceChecker;
var _configurationError = _interopRequireDefault(require("./configurationError.js"));
var _isSingleLineString = _interopRequireDefault(require("./isSingleLineString.js"));
var _isWhitespace = _interopRequireDefault(require("./isWhitespace.js"));
var _validateTypes = require("./validateTypes.js");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
function ownKeys(e, r) { var t = Object.keys(e); if (Object.getOwnPropertySymbols) { var o = Object.getOwnPropertySymbols(e); r && (o = o.filter(function (r) { return Object.getOwnPropertyDescriptor(e, r).enumerable; })), t.push.apply(t, o); } return t; }
function _objectSpread(e) { for (var r = 1; r < arguments.length; r++) { var t = null != arguments[r] ? arguments[r] : {}; r % 2 ? ownKeys(Object(t), !0).forEach(function (r) { _defineProperty(e, r, t[r]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(e, Object.getOwnPropertyDescriptors(t)) : ownKeys(Object(t)).forEach(function (r) { Object.defineProperty(e, r, Object.getOwnPropertyDescriptor(t, r)); }); } return e; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return _typeof(key) === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (_typeof(input) !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (_typeof(res) !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
/**
 * @typedef {(message: string) => string} MessageFunction
 */

/**
 * @typedef {Object} Messages
 * @property {MessageFunction} [expectedBefore]
 * @property {MessageFunction} [rejectedBefore]
 * @property {MessageFunction} [expectedAfter]
 * @property {MessageFunction} [rejectedAfter]
 * @property {MessageFunction} [expectedBeforeSingleLine]
 * @property {MessageFunction} [rejectedBeforeSingleLine]
 * @property {MessageFunction} [expectedBeforeMultiLine]
 * @property {MessageFunction} [rejectedBeforeMultiLine]
 * @property {MessageFunction} [expectedAfterSingleLine]
 * @property {MessageFunction} [rejectedAfterSingleLine]
 * @property {MessageFunction} [expectedAfterMultiLine]
 * @property {MessageFunction} [rejectedAfterMultiLine]
 */

/**
 * @typedef {Object} WhitespaceCheckerArgs
 * @property {string} source - The source string
 * @property {number} index - The index of the character to check before
 * @property {(message: string) => void} err - If a problem is found, this callback
 *   will be invoked with the relevant warning message.
 *   Typically this callback will report() the problem.
 * @property {string} [errTarget] - If a problem is found, this string
 *   will be sent to the relevant warning message.
 * @property {string} [lineCheckStr] - Single- and multi-line checkers
 *   will use this string to determine whether they should proceed,
 *   i.e. if this string is one line only, single-line checkers will check,
 *   multi-line checkers will ignore.
 *   If none is passed, they will use `source`.
 * @property {boolean} [onlyOneChar=false] - Only check *one* character before.
 *   By default, "always-*" checks will look for the `targetWhitespace` one
 *   before and then ensure there is no whitespace two before. This option
 *   bypasses that second check.
 * @property {boolean} [allowIndentation=false] - Allow arbitrary indentation
 *   between the `targetWhitespace` (almost definitely a newline) and the `index`.
 *   With this option, the checker will see if a newline *begins* the whitespace before
 *   the `index`.
 */

/**
 * @typedef {(args: WhitespaceCheckerArgs) => void} WhitespaceChecker
 */

/**
 * @typedef {{
 *   before: WhitespaceChecker,
 *   beforeAllowingIndentation: WhitespaceChecker,
 *   after: WhitespaceChecker,
 *   afterOneOnly: WhitespaceChecker,
 * }} WhitespaceCheckers
 */
/**
 * Create a whitespaceChecker, which exposes the following functions:
 * - `before()`
 * - `beforeAllowingIndentation()`
 * - `after()`
 * - `afterOneOnly()`
 *
 * @param {"space" | "newline"} targetWhitespace - This is a keyword instead
 *   of the actual character (e.g. " ") in order to accommodate
 *   different styles of newline ("\n" vs "\r\n")
 * @param {"always" | "never" | "always-single-line" | "always-multi-line" | "never-single-line" | "never-multi-line"} expectation
 * @param {Messages} messages - An object of message functions;
 *   calling `before*()` or `after*()` and the `expectation` that is passed
 *   determines which message functions are required
 *
 * @returns {WhitespaceCheckers} The checker, with its exposed checking functions
 */
function whitespaceChecker(targetWhitespace, expectation, messages) {
  // Keep track of active arguments in order to avoid passing
  // too much stuff around, making signatures long and confusing.
  // This variable gets reset anytime a checking function is called.
  /** @type {WhitespaceCheckerArgs} */
  var activeArgs;

  /**
   * Check for whitespace *before* a character.
   * @type {WhitespaceChecker}
   */
  function before(_ref) {
    var source = _ref.source,
      index = _ref.index,
      err = _ref.err,
      errTarget = _ref.errTarget,
      lineCheckStr = _ref.lineCheckStr,
      _ref$onlyOneChar = _ref.onlyOneChar,
      onlyOneChar = _ref$onlyOneChar === void 0 ? false : _ref$onlyOneChar,
      _ref$allowIndentation = _ref.allowIndentation,
      allowIndentation = _ref$allowIndentation === void 0 ? false : _ref$allowIndentation;
    activeArgs = {
      source: source,
      index: index,
      err: err,
      errTarget: errTarget,
      onlyOneChar: onlyOneChar,
      allowIndentation: allowIndentation
    };
    switch (expectation) {
      case "always":
        expectBefore();
        break;
      case "never":
        rejectBefore();
        break;
      case "always-single-line":
        if (!(0, _isSingleLineString["default"])(lineCheckStr || source)) {
          return;
        }
        expectBefore(messages.expectedBeforeSingleLine);
        break;
      case "never-single-line":
        if (!(0, _isSingleLineString["default"])(lineCheckStr || source)) {
          return;
        }
        rejectBefore(messages.rejectedBeforeSingleLine);
        break;
      case "always-multi-line":
        if ((0, _isSingleLineString["default"])(lineCheckStr || source)) {
          return;
        }
        expectBefore(messages.expectedBeforeMultiLine);
        break;
      case "never-multi-line":
        if ((0, _isSingleLineString["default"])(lineCheckStr || source)) {
          return;
        }
        rejectBefore(messages.rejectedBeforeMultiLine);
        break;
      default:
        throw (0, _configurationError["default"])("Unknown expectation \"".concat(expectation, "\""));
    }
  }

  /**
   * Check for whitespace *after* a character.
   * @type {WhitespaceChecker}
   */
  function after(_ref2) {
    var source = _ref2.source,
      index = _ref2.index,
      err = _ref2.err,
      errTarget = _ref2.errTarget,
      lineCheckStr = _ref2.lineCheckStr,
      _ref2$onlyOneChar = _ref2.onlyOneChar,
      onlyOneChar = _ref2$onlyOneChar === void 0 ? false : _ref2$onlyOneChar;
    activeArgs = {
      source: source,
      index: index,
      err: err,
      errTarget: errTarget,
      onlyOneChar: onlyOneChar
    };
    switch (expectation) {
      case "always":
        expectAfter();
        break;
      case "never":
        rejectAfter();
        break;
      case "always-single-line":
        if (!(0, _isSingleLineString["default"])(lineCheckStr || source)) {
          return;
        }
        expectAfter(messages.expectedAfterSingleLine);
        break;
      case "never-single-line":
        if (!(0, _isSingleLineString["default"])(lineCheckStr || source)) {
          return;
        }
        rejectAfter(messages.rejectedAfterSingleLine);
        break;
      case "always-multi-line":
        if ((0, _isSingleLineString["default"])(lineCheckStr || source)) {
          return;
        }
        expectAfter(messages.expectedAfterMultiLine);
        break;
      case "never-multi-line":
        if ((0, _isSingleLineString["default"])(lineCheckStr || source)) {
          return;
        }
        rejectAfter(messages.rejectedAfterMultiLine);
        break;
      default:
        throw (0, _configurationError["default"])("Unknown expectation \"".concat(expectation, "\""));
    }
  }

  /**
   * @type {WhitespaceChecker}
   */
  function beforeAllowingIndentation(obj) {
    before(_objectSpread(_objectSpread({}, obj), {}, {
      allowIndentation: true
    }));
  }
  function expectBefore() {
    var messageFunc = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : messages.expectedBefore;
    if (activeArgs.allowIndentation) {
      expectBeforeAllowingIndentation(messageFunc);
      return;
    }
    var _activeArgs = activeArgs;
    var source = _activeArgs.source;
    var index = _activeArgs.index;
    var oneCharBefore = source[index - 1];
    var twoCharsBefore = source[index - 2];
    if ((0, _validateTypes.isNullish)(oneCharBefore)) {
      return;
    }
    if (targetWhitespace === "space" && oneCharBefore === " " && (activeArgs.onlyOneChar || (0, _validateTypes.isNullish)(twoCharsBefore) || !(0, _isWhitespace["default"])(twoCharsBefore))) {
      return;
    }
    (0, _validateTypes.assertFunction)(messageFunc);
    activeArgs.err(messageFunc(activeArgs.errTarget || source.charAt(index)));
  }
  function expectBeforeAllowingIndentation() {
    var messageFunc = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : messages.expectedBefore;
    var _activeArgs2 = activeArgs;
    var source = _activeArgs2.source;
    var index = _activeArgs2.index;
    var err = _activeArgs2.err;
    var expectedChar = targetWhitespace === "newline" ? "\n" : undefined;
    var i = index - 1;
    while (source[i] !== expectedChar) {
      if (source[i] === "\t" || source[i] === " ") {
        i--;
        continue;
      }
      (0, _validateTypes.assertFunction)(messageFunc);
      err(messageFunc(activeArgs.errTarget || source.charAt(index)));
      return;
    }
  }
  function rejectBefore() {
    var messageFunc = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : messages.rejectedBefore;
    var _activeArgs3 = activeArgs;
    var source = _activeArgs3.source;
    var index = _activeArgs3.index;
    var oneCharBefore = source[index - 1];
    if (!(0, _validateTypes.isNullish)(oneCharBefore) && (0, _isWhitespace["default"])(oneCharBefore)) {
      (0, _validateTypes.assertFunction)(messageFunc);
      activeArgs.err(messageFunc(activeArgs.errTarget || source.charAt(index)));
    }
  }

  /**
   * @type {WhitespaceChecker}
   */
  function afterOneOnly(obj) {
    after(_objectSpread(_objectSpread({}, obj), {}, {
      onlyOneChar: true
    }));
  }
  function expectAfter() {
    var messageFunc = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : messages.expectedAfter;
    var _activeArgs4 = activeArgs;
    var source = _activeArgs4.source;
    var index = _activeArgs4.index;
    var oneCharAfter = source[index + 1];
    var twoCharsAfter = source[index + 2];
    var threeCharsAfter = source[index + 3];
    if ((0, _validateTypes.isNullish)(oneCharAfter)) {
      return;
    }
    if (targetWhitespace === "newline") {
      // If index is followed by a Windows CR-LF ...
      if (oneCharAfter === "\r" && twoCharsAfter === "\n" && (activeArgs.onlyOneChar || (0, _validateTypes.isNullish)(threeCharsAfter) || !(0, _isWhitespace["default"])(threeCharsAfter))) {
        return;
      }

      // If index is followed by a Unix LF ...
      if (oneCharAfter === "\n" && (activeArgs.onlyOneChar || (0, _validateTypes.isNullish)(twoCharsAfter) || !(0, _isWhitespace["default"])(twoCharsAfter))) {
        return;
      }
    }
    if (targetWhitespace === "space" && oneCharAfter === " " && (activeArgs.onlyOneChar || (0, _validateTypes.isNullish)(twoCharsAfter) || !(0, _isWhitespace["default"])(twoCharsAfter))) {
      return;
    }
    (0, _validateTypes.assertFunction)(messageFunc);
    activeArgs.err(messageFunc(activeArgs.errTarget || source.charAt(index)));
  }
  function rejectAfter() {
    var messageFunc = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : messages.rejectedAfter;
    var _activeArgs5 = activeArgs;
    var source = _activeArgs5.source;
    var index = _activeArgs5.index;
    var oneCharAfter = source[index + 1];
    if (!(0, _validateTypes.isNullish)(oneCharAfter) && (0, _isWhitespace["default"])(oneCharAfter)) {
      (0, _validateTypes.assertFunction)(messageFunc);
      activeArgs.err(messageFunc(activeArgs.errTarget || source.charAt(index)));
    }
  }
  return {
    before: before,
    beforeAllowingIndentation: beforeAllowingIndentation,
    after: after,
    afterOneOnly: afterOneOnly
  };
}