# SOLID/SoC Refactoring Documentation

## Overview
This document outlines the SOLID principles and Separation of Concerns (SoC) refactoring applied to the Kiqro Web project.

## Principles Applied

### 1. Single Responsibility Principle (SRP)
- Each data file has a single, well-defined responsibility
- Pages are responsible only for UI rendering and user interaction
- Business logic is separated from presentation logic

### 2. Separation of Concerns (SoC)
- **Data Layer** (`/src/data`): All static/mock data
- **Business Logic** (`/src/hooks`): Custom hooks for state management (future)
- **Configuration** (`/src/constants`): Application constants and configs (future)
- **Presentation** (`/src/app`, `/src/components`): UI components and pages

## Directory Structure

```
src/
├── data/                    # Data Layer - SOLID Principle: SRP
│   ├── index.ts            # Central export point
│   ├── tasks.ts            # Task-related data (ongoing, completed, withdrawals)
│   ├── macroTasks.ts       # Macro task data with categories
│   ├── microTasks.ts       # Micro task data with badges
│   ├── offers.ts           # Offers data (banking, tech, partners)
│   └── leaderboard.ts      # Leaderboard users and rewards
│
├── hooks/                  # Business Logic Layer (future)
│   └── (custom hooks will go here)
│
├── constants/              # Configuration Layer (future)
│   └── (constants will go here)
│
├── app/                    # Presentation Layer
└── components/             # Presentation Layer
```

## Data Layer Structure

### 1. Tasks Data (`/src/data/tasks.ts`)
**Responsibility**: Manage all task-related data

**Exports**:
- `ongoingTasks`: Array of ongoing tasks
- `completedTasks`: Array of completed tasks
- `withdrawalTasks`: Array of withdrawal transactions

**Type**: Imports `Task` type from `TaskTable` component

**Usage**:
```typescript
import { ongoingTasks, completedTasks, withdrawalTasks } from '@/data';
```

### 2. Macro Tasks Data (`/src/data/macroTasks.ts`)
**Responsibility**: Manage macro task offers with categories

**Interface**:
```typescript
interface MacroTask {
  id: string;
  title: string;
  subtitle: string;
  amount: string;
  image: string;
  imageAlt: string;
  category: 'banking' | 'tech' | 'survey' | 'social';
}
```

**Exports**:
- `MacroTask` interface
- `allMacroTasks`: Array of 7 macro tasks across 4 categories

**Usage**:
```typescript
import { allMacroTasks, type MacroTask } from '@/data';
```

### 3. Micro Tasks Data (`/src/data/microTasks.ts`)
**Responsibility**: Manage micro task data with badge system

**Interface**:
```typescript
interface MicroTask {
  id: string;
  platform: string;
  platformIcon: string;
  title: string;
  description: string;
  reward: string;
  completionRate: string;
  totalSlots: number;
  remainingSlots: number;
  badges?: {
    icon: IconName;
    value: string;
  }[];
}
```

**Exports**:
- `MicroTask` interface
- `microTasks`: Array of 4 Instagram tasks with completion tracking

**Type Safety**: Uses `IconName` type from `@/lib/icons` for badge icons

**Usage**:
```typescript
import { microTasks, type MicroTask } from '@/data';
```

### 4. Offers Data (`/src/data/offers.ts`)
**Responsibility**: Manage all offer-related data

**Interface**:
```typescript
interface OfferCardData {
  id: string;
  title: string;
  subtitle: string;
  amount: string;
  image: string;
  imageAlt: string;
  rating?: number;
}
```

**Exports**:
- `OfferCardData` interface
- `bankingOffers`: Banking offers (2 items)
- `techOffers`: Technology offers (1 item)
- `taskPartners`: Task partner offers (7 items with ratings)
- `surveyPartners`: Survey partner offers (6 items with ratings)

**Usage**:
```typescript
import { bankingOffers, techOffers, taskPartners, surveyPartners } from '@/data';
```

### 5. Leaderboard Data (`/src/data/leaderboard.ts`)
**Responsibility**: Manage leaderboard rankings and rewards

**Interfaces**:
```typescript
interface LeaderboardUser {
  rank: number;
  name: string;
  earnings: string;
}

interface LeaderboardReward {
  rank: string;
  amount: string;
}
```

**Exports**:
- `LeaderboardUser` interface
- `LeaderboardReward` interface
- `leaderboardUsers`: Array of 10 top users
- `leaderboardRewards`: Array of 5 reward tiers

**Usage**:
```typescript
import { leaderboardUsers, leaderboardRewards } from '@/data';
```

### 6. Central Export (`/src/data/index.ts`)
**Responsibility**: Provide single import point for all data

Re-exports all data modules:
```typescript
export * from './tasks'
export * from './macroTasks'
export * from './microTasks'
export * from './offers'
export * from './leaderboard'
```

**Usage**:
```typescript
// Import everything from central location
import { ongoingTasks, allMacroTasks, microTasks, bankingOffers } from '@/data';
```

## Refactored Pages

### 1. Home Page (`/app/(dashboard)/home/page.tsx`)
**Before**: 280+ lines with inline data arrays
**After**: Clean imports from data layer

```typescript
import { bankingOffers, techOffers, taskPartners, surveyPartners } from '@/data';
```

**Benefit**: Reduced code by ~200 lines, improved maintainability

### 2. Macro Tasks Page (`/app/(dashboard)/macro-tasks/page.tsx`)
**Before**: 150+ lines with inline MacroTask array
**After**: Single import statement

```typescript
import { allMacroTasks } from '@/data';
```

**Benefit**: Reduced code by ~100 lines, centralized task data

### 3. Micro Tasks Page (`/app/(dashboard)/micro-tasks/page.tsx`)
**Before**: 70+ lines with inline task definitions
**After**: Clean import with proper typing

```typescript
import { microTasks as initialMicroTasks } from '@/data';
```

**Benefit**: Reduced code by ~50 lines, improved type safety

### 4. Profile Page (`/app/(dashboard)/profile/page.tsx`)
**Before**: 180+ lines with three large task arrays
**After**: Simple imports from data layer

```typescript
import { ongoingTasks, completedTasks, withdrawalTasks } from '@/data';
```

**Benefit**: Reduced code by ~150 lines, eliminated data duplication

## Benefits Achieved

### 1. Code Reduction
- **Total lines removed**: ~500+ lines of inline data definitions
- **Improved readability**: Pages are now focused on UI logic only
- **Easier maintenance**: Data changes require updates in one place only

### 2. Type Safety
- All data exports have proper TypeScript interfaces
- Type inference works correctly across the application
- IconName type ensures badge icons are valid

### 3. Reusability
- Same data can be imported in multiple components
- No duplication of mock data
- Easy to swap with API calls later

### 4. Testing
- Data layer can be tested independently
- Easy to mock data for component tests
- Clear separation between data and presentation

### 5. Future-Proofing
- Easy to replace with API calls
- Hook layer ready for business logic
- Constants folder ready for configuration

## Migration Guide

### How to Add New Data

1. **Create a new data file** in `/src/data/`:
```typescript
// /src/data/newFeature.ts
export interface NewFeatureData {
  id: string;
  name: string;
}

export const newFeatures: NewFeatureData[] = [
  { id: '1', name: 'Feature 1' }
];
```

2. **Export from index.ts**:
```typescript
// /src/data/index.ts
export * from './newFeature'
```

3. **Use in components**:
```typescript
import { newFeatures } from '@/data';
```

### How to Convert API Data

When ready to use real APIs:

1. **Create a hook** in `/src/hooks/`:
```typescript
// /src/hooks/useTasks.ts
export function useTasks() {
  const [tasks, setTasks] = useState([]);
  
  useEffect(() => {
    fetch('/api/tasks')
      .then(res => res.json())
      .then(setTasks);
  }, []);
  
  return { tasks };
}
```

2. **Replace data imports** with hook:
```typescript
// Before
import { ongoingTasks } from '@/data';

// After
import { useTasks } from '@/hooks/useTasks';
const { tasks: ongoingTasks } = useTasks();
```

## Best Practices

1. **Keep data files focused**: One concern per file
2. **Use TypeScript interfaces**: Always define types
3. **Export from index.ts**: Maintain central export point
4. **Document data structure**: Add comments for complex data
5. **Follow naming conventions**: Use descriptive, plural names for arrays

## Next Steps

### Phase 2: Hooks Layer
- [ ] Create `useTasks` hook for task management
- [ ] Create `useLeaderboard` hook for leaderboard data
- [ ] Create `useOffers` hook for offers management
- [ ] Create `useAuth` hook for authentication

### Phase 3: Constants Layer
- [ ] Move category definitions to constants
- [ ] Create navigation configuration
- [ ] Define status mappings
- [ ] Create filter configurations

### Phase 4: API Integration
- [ ] Replace data imports with API calls
- [ ] Implement data fetching hooks
- [ ] Add loading and error states
- [ ] Implement caching strategies

## Conclusion

The SOLID/SoC refactoring has successfully separated data from presentation logic, creating a more maintainable and scalable codebase. The project now follows clean architecture principles with clear boundaries between layers.

**Key Achievement**: Reduced codebase by ~500 lines while improving maintainability and type safety.

**Build Status**: ✅ All builds passing with only minor unused variable warnings

---

*Last Updated: March 2025*
*Author: GitHub Copilot*
