import Image from "next/image";

interface LeaderboardUser {
  rank: number;
  name: string;
  avatar: string;
  amount: string;
  badge?: "winner" | "second" | "third";
}

interface LeaderboardSectionProps {
  users: LeaderboardUser[];
  currentUserRank?: number;
}

export default function LeaderboardSection({ users, currentUserRank }: LeaderboardSectionProps) {
  // İlk 3'ü ayır
  const topThree = users.slice(0, 3);
  const restUsers = users.slice(3);

  // Kazancı badge'e göre sırala (2, 1, 3 şeklinde gösterim için)
  const podium = [topThree[1], topThree[0], topThree[2]].filter(Boolean);

  return (
    <div className="bg-base-2 rounded-2xl md:rounded-3xl p-4 md:p-8 space-y-4 md:space-y-6">
      {/* Başlık */}
      <div className="flex  relative items-center justify-center">
        <div className="flex items-center justify-center flex-col">
          <h2 className="text-sm md:text-xl font-normal text-primary mb-1 text-center">
            Ödüllerin Verileceği Tarih
          </h2>
          <p className="text-lg md:text-2xl font-bold bg-white bg-clip-text text-transparent">
            31 Aralık 23:59
          </p>
        </div>
        <div className="absolute right-0 top-0  transform md:translate-y-0 translate-y-[160%] md:top-0 bg-primary rounded-full px-2 md:px-4 py-0.5 md:py-1">
          <span className="text-white font-medium text-xs md:text-sm">Sıran: {currentUserRank || 6}</span>
        </div>
      </div>

      {/* Podium (İlk 3) */}
      <div className="flex items-end justify-center gap-4 md:gap-8 py-4 md:py-6 px-2 md:px-4 overflow-hidden">
        {podium.map((user) => {
          if (!user) return null;
          const actualRank = user.rank as 1 | 2 | 3;
          const heights: Record<1 | 2 | 3, string> = { 2: 'h-24 md:h-32', 1: 'h-32 md:h-40', 3: 'h-20 md:h-28' };
          const scales: Record<1 | 2 | 3, string> = { 2: 'scale-90 md:scale-90', 1: 'scale-95 md:scale-100', 3: 'scale-85 md:scale-85' };

          return (
            <div key={user.rank} className={`flex pt-5 flex-col items-center gap-1 md:gap-1 ${scales[actualRank]} flex-shrink-0`}>
              {/* Avatar & Badge */}
              <div className="relative">
                {actualRank === 1 && (
                  <div className="absolute -top-6 md:-top-8 left-1/2 -translate-x-1/2">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" className="md:w-8 md:h-8">
                      <path d="M12 2L15 8.5L22 9.27L17 14.14L18.18 21.02L12 17.77L5.82 21.02L7 14.14L2 9.27L9 8.5L12 2Z" fill="#FFD700" stroke="#FFA500" strokeWidth="1.5"/>
                    </svg>
                  </div>
                )}
                <div className="relative">
                  <Image
                    src={user.avatar}
                    alt={user.name}
                    width={80}
                    height={80}
                    className="rounded-full border-2 md:border-4 border-base-3 w-14 h-14 md:w-20 md:h-20"
                  />
                  <div className={`absolute -bottom-1.5 md:-bottom-2 left-1/2 -translate-x-1/2 w-6 h-6 md:w-8 md:h-8 rounded-full flex items-center justify-center text-white font-bold text-sm md:text-lg ${
                    actualRank === 1 ? 'bg-yellow-500' : actualRank === 2 ? 'bg-gray-400' : 'bg-orange-500'
                  }`}>
                    {actualRank}
                  </div>
                </div>
              </div>

              {/* Name & Amount */}
              <div className="text-center ">
                <p className="text-white font-semibold text-xs md:text-base mb-0.5 md:mb-1 truncate max-w-[80px] md:max-w-none">{user.name}</p>
                <p className="text-white/90 font-medium text-xs md:text-sm">{user.amount}</p>
              </div>

              {/* Podium Stand - SVG */}
              <div className={`${heights[actualRank]} w-24 md:w-36 relative flex items-end justify-center`}>
                <svg 
                  viewBox="0 0 147 137" 
                  fill="none" 
                  xmlns="http://www.w3.org/2000/svg"
                  className="w-full h-full"
                  preserveAspectRatio="none"
                >
                  <rect x="0.826172" y="30.6042" width="145.559" height="106.395" fill={`url(#paint0_linear_rank_${actualRank})`}/>
                  <path d="M26.9354 0.578522H120.276L146.385 30.6042H0.826172L26.9354 0.578522Z" fill={`url(#paint1_linear_rank_${actualRank})`}/>
                  
                  {/* Sıra Numarası - Orta */}
                  <text
                    x="73.6"
                    y="95"
                    textAnchor="middle"
                    dominantBaseline="middle"
                    fill="#fff"
                    fontSize="80"
                    fontWeight="700"
                    opacity="0.1"
                  >
                    {actualRank}
                  </text>
                  
                  <defs>
                    <linearGradient id={`paint0_linear_rank_${actualRank}`} x1="73.6057" y1="30.6042" x2="73.6057" y2="136.999" gradientUnits="userSpaceOnUse">
                      <stop offset="0.408654" stopColor="#483A74"/>
                      <stop offset="0.682692" stopColor="#483A74" stopOpacity="0.4"/>
                      <stop offset="1" stopColor="#292242" stopOpacity="0"/>
                    </linearGradient>
                    <linearGradient id={`paint1_linear_rank_${actualRank}`} x1="73.6057" y1="0.578522" x2="73.6057" y2="30.6042" gradientUnits="userSpaceOnUse">
                      <stop stopColor="#28233F" stopOpacity="0"/>
                      <stop offset="1" stopColor="#6A5BA5"/>
                    </linearGradient>
                  </defs>
                </svg>
              </div>
            </div>
          );
        })}
      </div>

      {/* Kazancı ve Ödül Başlıkları */}
      <div className="hidden md:flex items-center justify-between space-x-12 px-4 py-1 text-gray-400 text-sm font-medium border-b border-base-3">
        <span className="flex-1">Kullanıcı Adı</span>
        <span className="w-24 text-right">Kazancı</span>
        <span className="w-24 text-right">Ödül</span>
      </div>

      {/* Diğer Kullanıcılar (4-10) */}
      <div className="space-y-2 max-h-64 overflow-y-auto custom-scrollbar">
        {restUsers.map((user) => (
          <div
            key={user.rank}
            className={`flex items-center justify-between gap-2 md:gap-4 py-2 md:py-1 px-2 md:px-4 rounded-xl transition ${
              user.rank === currentUserRank
                ? 'bg-purple-500/20 border border-purple-500/50'
                : 'bg-base-3 hover:bg-base-3/80'
            }`}
          >
            <div className="flex items-center gap-2 md:gap-3 flex-1 min-w-0">
              <span className="text-white font-medium text-sm md:text-base w-4 md:w-5 flex-shrink-0">{user.rank}</span>
              <Image
                src={user.avatar}
                alt={user.name}
                width={40}
                height={40}
                className="rounded-full w-8 h-8 md:w-10 md:h-10 flex-shrink-0"
              />
              <span className="text-white font-medium text-xs md:text-base truncate">{user.name}</span>
            </div>
            <span className="text-white font-semibold text-xs md:text-sm text-right flex-shrink-0">{user.amount}</span>
            <span className="text-green-400 text-[10px] md:text-sm font-medium w-16 md:w-24 text-right flex-shrink-0">Çeyrek Altın</span>
          </div>
        ))}
      </div>
    </div>
  );
}
