'use client'

import { Icon } from "@/components/ui";
import Image from "next/image";

export interface Task {
  id: string;
  logo: string;
  title: string;
  description: string;
  amount: string;
  status: string;
  date: string;
}

export interface TaskTableProps {
  /** Görev listesi */
  tasks: Task[];
  /** Tablo başlıkları (opsiyonel, varsayılan değerler kullanılır) */
  headers?: {
    taskName?: string;
    description?: string;
    amount?: string;
    status?: string;
    date?: string;
  };
  /** Boş durumda gösterilecek mesaj */
  emptyMessage?: string;
  /** Ek CSS sınıfları */
  className?: string;
}

export function TaskTable({ 
  tasks, 
  headers = {
    taskName: 'Görev Adı',
    amount: 'Tutar',
    status: 'Durum',
    date: 'Tarih'
  },
  emptyMessage = 'Henüz görev bulunmuyor',
  className = ''
}: TaskTableProps) {
  
  // 5 sütunlu mu yoksa 4 sütunlu mu kontrol et
  const hasSeparateDescription = headers.description !== undefined;
  const gridCols = hasSeparateDescription ? 'grid-cols-5' : 'grid-cols-4';
  
  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Devam Ediyor':
        return 'text-[#6A4DF5]';
      case 'Tamamlandı':
        return 'text-green-400';
      case 'İşleniyor':
        return 'text-yellow-400';
      case 'Onaylandı':
        return 'text-blue-400';
      case 'Gerçekleşti':
        return 'text-green-400';
      case 'Onay Bekliyor':
        return 'text-purple-400';
      case 'Red Edildi':
        return 'text-red-400';
      default:
        return 'text-gray-400';
    }
  };

  return (
    <div className={`h-auto md:h-[550px] rounded-2xl overflow-hidden ${className}`}>
      {/* Tablo Başlıkları */}
      <div className={`hidden lg:grid text-sm font-normal ${gridCols} gap-4 py-4 border-b border-gray-700`}>
        <div className="text-gray-400">{headers.taskName}</div>
        {hasSeparateDescription && <div className="text-gray-400 text-center">{headers.description}</div>}
        <div className="text-gray-400 text-center">{headers.date}</div>
        <div className="text-gray-400 text-center">{headers.amount}</div>
        <div className="text-gray-400 text-center">{headers.status}</div>
      </div>

      {/* Görev Satırları */}
      <div className="divide-y overflow-y-auto max-h-full divide-gray-700">
        {tasks.map((task) => (
          <div
            key={task.id}
            className={`grid grid-cols-1 lg:${gridCols} gap-2 lg:gap-4 p-4 md:p-6 hover:bg-[#221C3A] transition cursor-pointer`}
          >
            {/* Görev Adı */}
            <div className="flex items-center gap-3 md:gap-4 mb-2 lg:mb-0">
              {/* Logo veya Icon gösterimi */}
              {task.logo === 'papara' ? (
                <div className="w-10 h-10 md:w-12 md:h-12 rounded-lg bg-base-2 flex items-center justify-center">
                  <Icon name="papara" size={32} />
                </div>
              ) : task.logo === 'usdc' ? (
                <div className="w-10 h-10 md:w-12 md:h-12 rounded-lg bg-base-2 flex items-center justify-center">
                  <Icon name="token-branded:usdt" className="text-3xl" />
                </div>
              ) : task.logo === 'bank' ? (
                <div className="w-10 h-10 md:w-12 md:h-12 rounded-lg bg-base-2 flex items-center justify-center">
                  <Icon name="mdi:bank-outline" className="text-3xl text-white" />
                </div>
              ) : (
                <Image
                  src={task.logo}
                  alt={task.title}
                  width={48}
                  height={48}
                  className="rounded-lg w-10 h-10 md:w-12 md:h-12"
                  unoptimized
                />
              )}
              <div className="flex-1 min-w-0">
                <h3 className="font-semibold text-white text-sm md:text-base truncate">{task.title}</h3>
                {!hasSeparateDescription && task.description && (
                  <p className="text-xs md:text-sm text-gray-400 truncate">{task.description}</p>
                )}
              </div>
            </div>

            {/* İşlem Numarası (sadece 5 sütunlu tabloda) */}
            {hasSeparateDescription && (
              <div className="flex justify-between lg:justify-center items-center">
                <span className="lg:hidden text-xs md:text-sm text-gray-400 mr-2 min-w-[100px]">{headers.description}:</span>
                <span className="text-gray-500 text-sm md:text-base">{task.description}</span>
              </div>
            )}

            {/* Tarih */}
            <div className="flex justify-between lg:justify-center items-center">
              <span className="lg:hidden text-xs md:text-sm text-gray-400 mr-2 min-w-[100px]">{headers.date}:</span>
              <span className="text-gray-400 font-medium text-sm md:text-base">{task.date}</span>
            </div>

            {/* Tutar */}
            <div className="flex justify-between lg:justify-center items-center">
              <span className="lg:hidden text-xs md:text-sm text-gray-400 mr-2 min-w-[100px]">{headers.amount}:</span>
              <span className=" text-gray-400 font-medium text-sm md:text-base">{task.amount}</span>
            </div>

            {/* Durum */}
            <div className="flex justify-between lg:justify-center items-center">
              <span className="lg:hidden text-xs md:text-sm text-gray-400 mr-2 min-w-[100px]">{headers.status}:</span>
              <span className={`font-medium text-sm md:text-base ${getStatusColor(task.status)}`}>
                {task.status}
              </span>
            </div>
          </div>
        ))}
      </div>

      {/* Boş Durum */}
      {tasks.length === 0 && (
        <div className="p-12 text-center">
          <Icon size="2xl" name="task-1" />
          <p className="text-gray-400 mt-4">{emptyMessage}</p>
        </div>
      )}
    </div>
  );
}

export default TaskTable;
