'use client'

import { useState, useMemo } from "react";
import { PlatformSelector } from "@/components/ui";
import { Footer } from "@/components/layout";
import Image from "next/image";
import { getLeaderboardData, getRewardAmount } from "@/data";
import { isArray } from '@/lib/guards';

export default function LeaderboardPage() {
  const [timeFilter, setTimeFilter] = useState<'daily' | 'monthly'>('daily');
  const currentUserRank = 6;

  const filterOptions = [
    { id: 'daily', label: 'Günlük', value: 'daily' },
    { id: 'monthly', label: 'Aylık', value: 'monthly' },
  ];

  // Validate data from data layer
  const leaderboardUsers = useMemo(() => {
    const data = getLeaderboardData();
    return isArray(data) && data.length > 0 ? data : [];
  }, []);

  // İlk 3'ü ayır
  const topThree = useMemo(() => {
    return leaderboardUsers.slice(0, 3);
  }, [leaderboardUsers]);

  const restUsers = useMemo(() => {
    return leaderboardUsers.slice(3);
  }, [leaderboardUsers]);

  // Podium sıralaması (2, 1, 3)
  const podium = useMemo(() => {
    return [topThree[1], topThree[0], topThree[2]].filter(Boolean);
  }, [topThree]);

  return (
    <div className="md:h-screen pt-20 md:pt-0 overflow-y-auto w-full p-3 md:p-6 lg:pt-20 lg:p-4">
      <div className=" p-4 md:p-8 flex flex-col items-center space-y-4 md:space-y-6">
        {/* Başlık */}
        <div className="flex relative items-center justify-center mb-2 md:mb-4">
          <div className="flex items-center justify-center flex-col">
            <h2 className="text-lg md:text-xl font-medium text-primary mb-1">
             Referans Liderlik Sıralaması
            </h2>
            <p className="text-xs md:text-sm text-gray-400">
              Ocak Tutarlarının Yatırılacağı Tarih
            </p>
            <p className="text-xl md:text-2xl font-bold text-white mt-1">
              02 : 12 : 24
            </p>
          </div>
        </div>

        {/* Filtre Butonları */}
        <div className="flex items-center  w-full justify-center mb-4 md:mb-6">
          <PlatformSelector
            options={filterOptions}
            defaultSelected={timeFilter}
            onChange={(value) => setTimeFilter(value as 'daily' | 'monthly')}
            size="md"
          />
        </div>

        {/* Empty State */}
        {leaderboardUsers.length === 0 ? (
          <div className="text-center py-16">
            <div className="text-gray-600 text-6xl mb-4">🏆</div>
            <p className="text-gray-400 text-lg mb-2">Henüz sıralama verisi yok</p>
            <p className="text-gray-500 text-sm">Liderlik tablosu yakında güncellenecek</p>
          </div>
        ) : (
          <>
            {/* Podium (İlk 3) */}
            <div className="flex  items-end justify-center gap-4 md:gap-8 py-4 md:py-6 px-2 md:px-4 w-full overflow-hidden">
          {podium.map((user) => {
            if (!user) return null;
            const actualRank = user.rank as 1 | 2 | 3;
            const heights: Record<1 | 2 | 3, string> = { 2: 'h-32 md:h-48', 1: 'h-40 md:h-56', 3: 'h-28 md:h-40' };
            const scales: Record<1 | 2 | 3, string> = { 2: 'scale-90 md:scale-95', 1: 'scale-100', 3: 'scale-85 md:scale-90' };

            return (
              <div key={user.rank} className={`flex pt-5 flex-col items-center gap-2  md:gap-3 ${scales[actualRank]} flex-shrink-0`}>
                {/* Avatar & Badge */}
                <div className="relative ">
                  {actualRank === 1 && (
                    <div className="absolute -top-6 md:-top-10 left-1/2 -translate-x-1/2">
                      <svg width="28" height="28" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" className="md:w-10 md:h-10">
                        <path d="M12 2L15 8.5L22 9.27L17 14.14L18.18 21.02L12 17.77L5.82 21.02L7 14.14L2 9.27L9 8.5L12 2Z" fill="#FFD700" stroke="#FFA500" strokeWidth="1.5"/>
                      </svg>
                    </div>
                  )}
                  <div className="relative">
                    <Image
                      src={user.avatar}
                      alt={user.name}
                      width={100}
                      height={100}
                      className="rounded-full border-2 md:border-4 border-base-3 w-16 h-16 md:w-24 md:h-24"
                    />
                    <div className={`absolute -bottom-2 md:-bottom-3 left-1/2 -translate-x-1/2 w-5 h-5 md:w-7 md:h-7 rounded-full flex items-center justify-center text-white font-bold text-base md:text-xl ${
                      actualRank === 1 ? 'bg-yellow-500' : actualRank === 2 ? 'bg-gray-400' : 'bg-orange-500'
                    }`}>
                      {actualRank}
                    </div>
                  </div>
                </div>

                {/* Name & Reward */}
                <div className="text-center">
                  <p className="text-white font-semibold text-sm md:text-lg mb-0.5 md:mb-1 truncate max-w-[80px] md:max-w-none">{user.name}</p>
                  <p className="text-white/90 font-medium text-xs md:text-base">{getRewardAmount(actualRank, timeFilter)}</p>
                </div>

              
               
                {/* Podium Stand - SVG */}
                <div className={`${heights[actualRank]} w-24   md:w-40 relative flex items-end justify-center`}>
                 
                     {/* Ödül Tutarı */}
                <div className="absolute text-[#A391E1] text-lg md:text-2xl font-bold mt-4 -bottom-5">
                  {getRewardAmount(actualRank, timeFilter)}
                </div>
                  <svg 
                    viewBox="0 0 147 137" 
                    fill="none" 
                    xmlns="http://www.w3.org/2000/svg"
                    className="w-full h-full"
                    preserveAspectRatio="none"
                  >
                    <rect x="0.826172" y="30.6042" width="145.559" height="106.395" fill={`url(#paint0_linear_rank_${actualRank})`}/>
                    <path d="M26.9354 0.578522H120.276L146.385 30.6042H0.826172L26.9354 0.578522Z" fill={`url(#paint1_linear_rank_${actualRank})`}/>
                    
                    {/* Sıra Numarası - Orta */}
                    <text
                      x="73"
                      y="95"
                      textAnchor="middle"
                      dominantBaseline="middle"
                      fill="#fff"
                      fontSize="60"
                      fontWeight="700"
                      opacity="0.1"
                      className="md:text-[70px]"
                    >
                      {actualRank}
                      
                    </text>

                    
                    
                    <defs>
                      <linearGradient id={`paint0_linear_rank_${actualRank}`} x1="73.6057" y1="30.6042" x2="73.6057" y2="136.999" gradientUnits="userSpaceOnUse">
                        <stop offset="0.408654" stopColor="#483A74"/>
                        <stop offset="0.682692" stopColor="#483A74" stopOpacity="0.4"/>
                        <stop offset="1" stopColor="#292242" stopOpacity="0"/>
                      </linearGradient>
                      <linearGradient id={`paint1_linear_rank_${actualRank}`} x1="73.6057" y1="0.578522" x2="73.6057" y2="30.6042" gradientUnits="userSpaceOnUse">
                        <stop stopColor="#28233F" stopOpacity="0"/>
                        <stop offset="1" stopColor="#6A5BA5"/>
                      </linearGradient>
                    </defs>
                  </svg>
                </div>
              </div>
            );
          })}
        </div>

        {/* Sıra Bilgisi */}
        <div className="flex items-center justify-center gap-2 mb-3 md:mb-4">
          <span className="text-gray-400 text-xs md:text-sm">Sıran:</span>
          <span className="text-white font-bold text-base md:text-lg">{currentUserRank}</span>
        </div>

        {/* Tablo Başlıkları - Hidden on mobile */}
        <div className="hidden md:grid grid-cols-[60px_1fr_120px] items-center px-4 py-3 text-gray-400 text-sm font-medium border-b border-base-3 w-full max-w-[1000px]">
          <span></span>
          <span>Kullanıcı Adı</span>
          <span className="text-right">Ödül</span>
        </div>

        {/* Kullanıcı Listesi (4-25) */}
        <div className="space-y-2 max-h-[400px] md:max-h-[500px] w-full max-w-[1000px] overflow-y-auto custom-scrollbar">
          {restUsers.map((user) => (
            <div
              key={user.rank}
              className={`grid grid-cols-[40px_1fr_80px] md:grid-cols-[60px_1fr_120px] items-center px-3 md:px-4 py-2 md:py-3 rounded-xl transition ${
                user.rank === currentUserRank
                  ? 'bg-purple-500/20 border border-purple-500/50'
                  : 'bg-base-3 hover:bg-base-3/80'
              }`}
            >
              <span className="text-white font-medium text-sm md:text-base">{user.rank}</span>
              <div className="flex items-center gap-2 md:gap-3 min-w-0">
                <Image
                  src={user.avatar}
                  alt={user.name}
                  width={40}
                  height={40}
                  className="rounded-full w-8 h-8 md:w-10 md:h-10 flex-shrink-0"
                />
                <span className="text-white font-medium text-xs md:text-base truncate">{user.name}</span>
              </div>
              <span className="text-green-400 text-xs md:text-sm font-medium text-right">
                {getRewardAmount(user.rank, timeFilter)}
              </span>
            </div>
          ))}
        </div>
          </>
        )}
      </div>
      
      <Footer />
    </div>
  );
}
