'use client'

import { useState, useMemo } from "react";
import { useRouter } from "next/navigation";
import { Iconify, Button, Icon, Pagination } from "@/components/ui";
import { Footer } from "@/components/layout";
import { usePagination } from "@/lib/hooks";
import Image from "next/image";
import { microTasks as initialMicroTasks } from '@/data';
import { isArray } from '@/lib/guards';

export default function MicroTasksPage() {
  const router = useRouter();
  
  // Validate and sanitize tasks data
  const tasks = useMemo(() => {
    return isArray(initialMicroTasks) && initialMicroTasks.length > 0 
      ? initialMicroTasks 
      : [];
  }, []);

  const [filteredTasks] = useState(tasks);

  // Pagination
  const {
    currentData,
    currentPage,
    totalPages,
    goToPage,
  } = usePagination({
    data: filteredTasks,
    itemsPerPage: 15,
  });

  const handleTaskClick = (taskId: string) => {
    if (taskId) {
      router.push(`/micro-tasks/${taskId}`);
    }
  };

  // Empty state check
  if (tasks.length === 0) {
    return (
      <div className="md:h-screen pt-20 md:pt-0 overflow-y-auto w-full p-3 md:p-6 lg:pt-20 lg:p-4">
        <div className="flex flex-col items-center justify-center h-[60vh] gap-4">
          <Iconify icon="mdi:clipboard-text-off-outline" size={64} className="text-gray-600" />
          <h2 className="text-xl font-semibold text-white">Henüz görev bulunmuyor</h2>
          <p className="text-gray-400 text-sm">Yeni görevler yakında eklenecek</p>
        </div>
        <Footer />
      </div>
    );
  }

  return (
    <div className="md:h-screen pt-20 md:pt-0 overflow-y-auto w-full p-3 md:p-6 lg:pt-20 lg:p-4">
      {/* Başlık */}
      <div className="mb-4 md:mb-6">
        <h1 className="text-xl md:text-2xl font-bold text-white mb-1 md:mb-2">Mikro Görevler</h1>
        <p className="text-gray-400 text-sm md:text-base">Aşağıdaki görevlerden birini seçerek hemen kazanmaya başlayın</p>
      </div>

      {/* Görev Listesi */}
      <div className="space-y-3 md:space-y-4">
        {currentData.map((task) => (
          <div
            key={task.id}
            className="bg-base-2 rounded-xl md:rounded-2xl py-3 md:py-4 px-3 md:px-4 flex flex-col md:flex-row items-start md:items-center justify-between hover:bg-base-2/80 transition-all duration-300 cursor-pointer group gap-3 md:gap-4"
          >
            {/* Sol Taraf - Platform & Bilgi */}
            <div className="flex items-start md:items-center gap-3 md:gap-4 flex-1 w-full">
              {/* Platform İkonu */}
              <Image
                src={task.platformIcon}
                alt={task.platform}
                width={400}
                height={400}
                className="object-cover rounded-xl md:rounded-2xl w-12 h-12 md:w-16 md:h-16 flex-shrink-0"
              />

              {/* Görev Bilgileri */}
              <div className="flex-1 min-w-0">
                <h3 className="text-white font-semibold text-sm md:text-lg mb-0.5 md:mb-1 group-hover:text-primary transition-colors line-clamp-2">
                  {task.title}
                </h3>
                <p className="text-gray-400 text-xs md:text-sm mb-1 md:mb-2 line-clamp-1">{task.description}</p>
              </div>
            </div>

            {/* Sağ Taraf - Tamamlanma, Badge & Buton */}
            <div className="flex items-center justify-between md:justify-start gap-3 md:gap-6 w-full md:w-auto">
              {/* Tamamlanma Oranı - Sabit genişlik */}
              <div className="flex items-center gap-1.5 md:w-28">
                <Icon name="mdi:user" className="text-gray-400" size={14} />
                <span className="text-gray-400 text-xs md:text-sm whitespace-nowrap">{task.completionRate}</span>
              </div>

              {/* Badge (Ödül/Puan) - Sabit genişlik */}
              <div className="flex items-center gap-2 md:w-32 justify-center flex-1 md:flex-initial">
                {task.badges?.map((badge, index) => (
                  <div
                    key={index}
                    className="flex items-center gap-1.5 md:gap-2"
                  >
                    <Icon name={badge.icon} className="text-primary" size={16} />
                    <span className="text-white text-sm md:text-base font-semibold whitespace-nowrap">{badge.value}</span>
                  </div>
                ))}
              </div>

              {/* Buton - Sabit genişlik */}
              <Button
                onClick={() => handleTaskClick(task.id)}
                variant="primary"
                size="md"
                className="md:w-40 flex-shrink-0"
                leftIcon={<Iconify icon="uil:link" size={18} />}
              >
                <span className="hidden md:inline">Göreve Git</span>
                <span className="md:hidden">Git</span>
              </Button>
            </div>
          </div>
        ))}
      </div>

      {/* Pagination */}
      <Pagination
        currentPage={currentPage}
        totalPages={totalPages}
        onPageChange={goToPage}
        className="mt-6 mb-8"
      />
      
      <Footer />
    </div>
  );
}
