'use client'

import { useState } from "react";
import { useRouter } from "next/navigation";
import { Button, Input, Iconify, Icon, Modal } from "@/components/ui";
import { StatCard } from "../../_components";
import Image from "next/image";
import { BANKS, type BankInfo } from "@/constants";
import { useWithdrawForm } from "@/hooks";

export default function BankWithdrawPage() {
  const router = useRouter();
  const [balance] = useState<string>("23.540₺");
  const [step, setStep] = useState<'bank-selection' | 'form'>('bank-selection');
  const [selectedBank, setSelectedBank] = useState<string | null>(null);
  const [showVerificationModal, setShowVerificationModal] = useState(false);

  const { formData, errors, updateField, setAmount, useAllBalance, handleSubmit } = useWithdrawForm({
    balance,
    onSubmit: () => setShowVerificationModal(true)
  });

  const handleBankSelect = (bankId: string) => {
    setSelectedBank(bankId);
    setStep('form');
  };

  const handleModalClose = () => {
    setShowVerificationModal(false);
  };

  const handleGoToSettings = () => {
    setShowVerificationModal(false);
    router.push('/settings');
  };

  const selectedBankData = BANKS.find(b => b.id === selectedBank);

  return (
    <div className="min-h-screen w-full mt-14 bg-base-1 p-4 lg:p-6">
      {/* Header */}
      <div className="w-full mx-auto mb-6">

        <div className="flex items-center gap-4">
          <div className="w-16 h-16 bg-base-2 rounded-2xl flex items-center justify-center">
            <Icon name="mdi:bank-outline" size={32} className="text-white" />
          </div>
          <div>
            <h1 className="text-2xl font-bold text-white mb-1">Havale / Eft</h1>
            <p className="text-gray-400 text-sm">%5 Çekim komisyonu</p>
          </div>
        </div>
      </div>
      <div className="w-full border-b border-white/10 mb-6" />

      {/* Çekilebilir Bakiye Kartı - Sadece form adımında göster */}
      {step === 'form' && (
        <div className="bg-base-2 rounded-3xl p-6 mb-6">
          <div className="flex items-center gap-3 mb-2">
            <Icon name="si:money-fill" size={24}/>
            <span className="text-gray-400 text-sm">Çekilebilir Bakiye</span>
          </div>
          <h2 className="text-4xl font-bold text-primary mb-4">{balance}</h2>
        </div>
      )}

      {/* Main Content */}
      <div className="w-full mx-auto grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Sol Taraf - Banka Seçimi veya Form */}
        <div className="bg-base-2 rounded-3xl p-6 lg:p-8">
          {step === 'bank-selection' ? (
            <>
              <h2 className="text-xl font-semibold text-white mb-6">Banka Seçimi</h2>
              <div className="space-y-3">
                {BANKS.map((bank: BankInfo) => (
                  <button
                    key={bank.id}
                    onClick={() => handleBankSelect(bank.id)}
                    className="w-full bg-base-3 hover:bg-base-3/80 rounded-2xl p-4 transition-all flex items-center justify-between group"
                  >
                    <div className="flex items-center gap-3">
                        <div className="w-24 h-12 bg-white rounded-xl overflow-hidden relative">
                        <Image
                          src={bank.logo}
                          alt={bank.name}
                          fill
                          className="object-cover  w-full h-full"
                          sizes="200px"
                        />
                        </div>
                      <span className="text-white font-medium">{bank.name}</span>
                    </div>
                    <Iconify 
                      icon="mdi:chevron-right" 
                      size={24} 
                      className="text-gray-400 group-hover:text-white transition-colors"
                    />
                  </button>
                ))}
              </div>
            </>
          ) : (
            <>
              <div className="flex items-center gap-3 mb-6">
                {selectedBankData && (
                  <>
                    <div className="w-24 relative h-12 bg-white rounded-xl flex items-center justify-center overflow-hidden">
                      <Image 
                        src={selectedBankData.logo} 
                        alt={selectedBankData.name}
                        fill
                        sizes="200px"
                        className="w-full h-full object-cover"
                      />
                    </div>
                    <h2 className="text-xl font-semibold text-white">{selectedBankData.name}</h2>
                  </>
                )}
              </div>

              <div className="space-y-5">
                {/* Hesap Adı */}
                <Input
                  label="Hesap Adı"
                  type="text"
                  placeholder="İsim Soyisim"
                  value={formData.accountName}
                  onChange={(e) => updateField('accountName', e.target.value)}
                  containerClassName="w-full"
                  error={errors.accountName}
                />

                {/* IBAN Numarası */}
                <Input
                  label="IBAN Numarası"
                  type="text"
                  placeholder="TR"
                  value={formData.iban}
                  onChange={(e) => updateField('iban', e.target.value)}
                  containerClassName="w-full"
                  error={errors.iban}
                />

                {/* Tutar Giriniz */}
                <div className="relative">
                  <Input
                    label="Tutar Giriniz"
                    type="text"
                    placeholder="Tutar"
                    value={formData.amount}
                    onChange={(e) => setAmount(e.target.value)}
                    containerClassName="w-full"
                    error={errors.amount}
                  />
                  <button
                    type="button"
                    onClick={useAllBalance}
                    className="absolute right-0 top-1 underline text-xs text-primary hover:text-primary/80 transition-colors font-medium"
                  >
                    Tüm Bakiye
                  </button>
                </div>

                {/* Tamam Butonu */}
                <div className="w-full flex justify-end">
                <Button
                  variant="primary"
                  size="lg"
                  onClick={() => handleSubmit('bank')}
                  className="mt-6 w-50"
                >
                  Tamam
                </Button>
                </div>
              </div>
            </>
          )}
        </div>

        {/* Sağ Taraf - Bilgilendirme */}
        <div className="bg-base-2 rounded-3xl p-6 lg:p-8">
          <h2 className="text-lg font-semibold text-primary mb-4">
            Banka EFT/HAVALE Çekim Talepleri Hakkında  Bilgilendirme
          </h2>

          <div className="space-y-4 text-gray-300 text-sm leading-relaxed">
            <p>
              Banka hesabınız üzerinden gerçekleştireceğiniz çekim taleplerinizde, işlemlerin yalnızca kendi adınıza kayıtlı banka hesap numaranıza yapılması zorunludur. Farklı bir kişi adına kayıtlı hesap numarasına çekim talebinde bulunulması durumunda, güvenlik sebebiyle işleminiz otomatik olarak reddedilecek ve tarafınıza kayıtlı e-posta adresinize bilgilendirme yapılacaktır.
            </p>

            <p>
              Çekim talepleriniz, yoğunluk durumuna bağlı olarak 24 saat içerisinde sonuçlandırılmaktadır. Sürecin sorunsuz ilerleyebilmesi için çekim talebi oluştururken banka hesabı bilgilerinizin doğru ve güncel olmasına özellikle dikkat etmeniz gerekmektedir. Bu sayede işlemleriniz zaman kaybı olmadan, güvenli bir şekilde tamamlanacaktır.
            </p>
          </div>
        </div>
      </div>

      {step === 'bank-selection' && (
        <div className="w-full mx-auto mt-8 justify-end flex">
          <Button
            variant="primary"
            size="lg"
            disabled
            className="min-w-[200px]"
          >
            Devam Et
          </Button>
        </div>
      )}

      {/* Verification Modal */}
      <Modal isOpen={showVerificationModal} onClose={handleModalClose}>
        <div className="p-6 text-center space-y-6">
          {/* Warning Icon */}
          <div className="flex justify-center">
            <StatCard
              icon="material-symbols:warning-outline"
              size="lg"
              color="default"
              iconColor="text-yellow-500"
            />
          </div>

          <div className="space-y-2">
            <p className="text-sm text-white/70">
              E-posta veya telefon doğrulaması yapmanız gerekiyor.
            </p>
          </div>
          
          <Button 
            variant="primary" 
            size="md" 
            fullWidth 
            onClick={handleGoToSettings}
          >
            Ayarlara Git
          </Button>
        </div>
      </Modal>
    </div>
  );
}
