'use client'

import { useState } from 'react'
import { Iconify } from '@/components/ui'

interface FAQItem {
  id: number
  question: string
  answer: string
}

const faqData: FAQItem[] = [
  {
    id: 1,
    question: "Kimler Kiqro'ya üye olmadan kullanabilir?",
    answer: "Kiqro platformuna üye olmadan erişim sağlayamazsınız. Platformumuzun tüm özelliklerinden faydalanabilmek için üyelik oluşturmanız gerekmektedir."
  },
  {
    id: 2,
    question: "Kiqro'ya nasıl üye olabilirim?",
    answer: "Kayıt sayfasından e-posta adresiniz ile hızlıca üye olabilirsiniz. Alternatif olarak Google veya Apple hesabınızla da giriş yapabilirsiniz."
  },
  {
    id: 3,
    question: "Görevleri tamamladıktan sonra ne kadar sürede ödeme alırım?",
    answer: "Görev onaylandıktan sonra ödemeniz hesabınıza 24-48 saat içerisinde yansıtılır. Ödemelerinizi cüzdanınızdan takip edebilirsiniz."
  },
  {
    id: 4,
    question: "Hangi ödeme yöntemlerini kullanabilirim?",
    answer: "Papara, Havale/EFT ve USDC gibi çeşitli ödeme yöntemlerini destekliyoruz. İstediğiniz yöntemi seçerek güvenli bir şekilde çekim yapabilirsiniz."
  },
  {
    id: 5,
    question: "Minimum çekim tutarı nedir?",
    answer: "Minimum çekim tutarı 50 TL'dir. Bu tutarın üzerinde olan bakiyenizi istediğiniz zaman çekebilirsiniz."
  },
  {
    id: 6,
    question: "Referans sistemi nasıl çalışır?",
    answer: "Referans linkinizi arkadaşlarınızla paylaşarak onların kazançlarından komisyon alabilirsiniz. Detaylı bilgi için Referanslar sayfasını ziyaret edebilirsiniz."
  }
]

interface FAQSectionProps {
  title?: string
  subtitle?: string
  maxItems?: number
  className?: string
}

export function FAQSection({
  title = 'Sıkça Sorulan Sorular',
  subtitle = 'Kiqro hakkında merak ettiğiniz her şey burada',
  maxItems = 6,
  className = ''
}: FAQSectionProps) {
  const [openItems, setOpenItems] = useState<number[]>([])

  const toggleItem = (id: number) => {
    if (openItems.includes(id)) {
      setOpenItems(openItems.filter(item => item !== id))
    } else {
      setOpenItems([...openItems, id])
    }
  }

  const displayedFAQs = faqData.slice(0, maxItems)

  return (
    <section className={`w-full py-12 md:py-16 ${className}`}>
      <div className="w-full max-w-5xl mx-auto px-4">
        {/* Header */}
        <div className="text-center mb-8 md:mb-12">
          <h2 className="text-2xl md:text-3xl lg:text-4xl font-bold mb-3 md:mb-4">
            {title}
          </h2>
          <p className="text-sm md:text-base lg:text-lg text-white/60 max-w-2xl mx-auto">
            {subtitle}
          </p>
        </div>

        {/* FAQ Items */}
        <div className="space-y-3 md:space-y-4">
          {displayedFAQs.map((item) => {
            const isOpen = openItems.includes(item.id)
            
            return (
              <div
                key={item.id}
                className="rounded-xl md:rounded-2xl border border-white/3  overflow-hidden transition-all hover:border-white/10"
              >
                {/* Question Button */}
                <button
                  onClick={() => toggleItem(item.id)}
                  className="w-full flex items-center justify-between gap-3 md:gap-4 p-4 md:p-6 text-left transition-colors"
                >
                  {/* Number Badge & Question */}
                  <div className="flex items-center gap-3 md:gap-4 flex-1 min-w-0">
                    <div className="w-10 h-10 md:w-12 md:h-12 bg-base-3 rounded-lg md:rounded-xl flex items-center justify-center flex-shrink-0">
                      <span className="text-white font-semibold text-sm md:text-lg">{item.id}</span>
                    </div>
                    
                    <h3 className="text-white text-sm md:text-base lg:text-lg font-medium leading-snug">
                      {item.question}
                    </h3>
                  </div>

                  {/* Toggle Icon */}
                  <div className="flex items-center gap-2 flex-shrink-0">
                    <span className="text-white/40 text-xs md:text-sm hidden sm:block">
                      {isOpen ? 'Gizle' : 'Görüntüle'}
                    </span>
                    <div className={`transform transition-transform duration-300 ${isOpen ? 'rotate-180' : ''}`}>
                      <Iconify 
                        icon="mdi:chevron-down" 
                        size={20} 
                        className="text-white/40 md:w-6 md:h-6"
                      />
                    </div>
                  </div>
                </button>

                {/* Answer */}
                <div
                  className={`transition-all duration-300 ease-in-out ${
                    isOpen ? 'max-h-96 opacity-100' : 'max-h-0 opacity-0'
                  } overflow-hidden`}
                >
                  <div className="px-4 md:px-6 pb-4 md:pb-6 pt-0">
                    <div className="pl-10 md:pl-16 pr-2 border-t border-white/5 pt-3 md:pt-4">
                      <p className="text-white/70 text-xs md:text-sm lg:text-base leading-relaxed">
                        {item.answer}
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            )
          })}
        </div>

        {/* Bottom CTA */}
        <div className="mt-8 md:mt-12 text-center">
          <p className="text-white/50 text-sm md:text-base mb-4">
            Sorunuz burada mı yok? Tüm soruları görün.
          </p>
          <a 
            href="/faq" 
            className="inline-flex items-center gap-2 bg-primary hover:bg-primary/90 text-white px-5 py-2.5 md:px-6 md:py-3 rounded-xl text-sm md:text-base font-medium transition-colors"
          >
            <span>Tüm SSS</span>
            <Iconify icon="mdi:arrow-right" size={20} />
          </a>
        </div>
      </div>
    </section>
  )
}
