'use client'

import React, { useState } from 'react'
import { Button, Input } from '@/components/ui'
import { Iconify } from '@/components/ui'

export interface SignupSectionProps {
  /** Ana başlık */
  mainTitle?: string
  /** Vurgulanan başlık kısmı */
  highlightedTitle?: string
  /** Açıklama metni */
  description?: string
  /** Kullanıcı sayısı */
  userCount?: string
  /** Form submit handler'ı */
  onFormSubmit?: (firstName: string, lastName: string, password: string) => void
  /** Google ile giriş handler'ı */
  onGoogleSignup?: () => void
  /** Apple ile giriş handler'ı */
  onAppleSignup?: () => void
  /** Facebook ile giriş handler'ı */
  onFacebookSignup?: () => void
  /** Son kayıt sayısı */
  recentSignups?: string
  /** Ek CSS sınıfları */
  className?: string
}

/**
 * SignupSection Component
 * Kayıt olma bölümü - sol tarafta başlık, sağ tarafta form
 */
const SignupSection: React.FC<SignupSectionProps> = ({
  mainTitle = "Şimdi katıl, dakikalar içinde",
  highlightedTitle = "kazanç fırsatını yakala!",
  description = "Binlerce kullanıcı gibi sen de hemen katıl, ödülleri anında kazanmaya başla.",
  userCount = "214,756 değerlendirmemizi görün",
  onFormSubmit,
  onGoogleSignup,
  onAppleSignup,
  onFacebookSignup,
  recentSignups = "77613+ son 24 saatteki üye kayıtları",
  className
}) => {
  const [firstName, setFirstName] = useState('')
  const [lastName, setLastName] = useState('')
  const [password, setPassword] = useState('')

  const handleFormSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    if (firstName.trim() && lastName.trim() && password.trim()) {
      if (onFormSubmit) {
        onFormSubmit(firstName.trim(), lastName.trim(), password.trim())
      } else {
        console.log('Ad:', firstName, 'Soyad:', lastName, 'Şifre:', password)
      }
    }
  }

  const handleGoogleSignup = () => {
    if (onGoogleSignup) {
      onGoogleSignup()
    } else {
      console.log('Google ile kayıt')
    }
  }

  const handleFacebookSignup = () => {
    if (onFacebookSignup) {
      onFacebookSignup()
    } else {
      console.log('Facebook ile kayıt')
    }
  }

  const handleAppleSignup = () => {
    if (onAppleSignup) {
      onAppleSignup()
    } else {
      console.log('Apple ile kayıt')
    }
  }

  return (
    <section className={`py-10 md:py-20 ${className}`}>
      <div className="container mx-auto pt-20 md:pt-40 px-4">
        <div className="flex flex-col lg:flex-row justify-between gap-8 md:gap-12 items-center">
          {/* Sol Taraf - Başlık ve Açıklama */}
          <div className="space-y-4 md:space-y-6 text-center lg:text-left w-full lg:w-auto">
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight">
              <span className="text-white">{mainTitle}</span>
              <br />
              <span className="text-[#BDB2FF]">{highlightedTitle}</span>
            </h2>
            
            <p className="text-gray-300 text-base md:text-lg leading-relaxed">
              {description}
            </p>

            {/* Kullanıcı Değerlendirmeleri */}
            <div className="flex items-center justify-center lg:justify-start space-x-2 text-gray-400">
              <span className="text-sm md:text-base">{userCount}</span>
              <Iconify 
                icon="solar:arrow-right-linear" 
                className="text-primary"
                size="sm"
              />
            </div>
          </div>

          {/* Sağ Taraf - Kayıt Formu */}
          <div className='bg-gradient-to-b p-[1px] from-[#8570ff86] to-transparent rounded-[25px] w-full lg:w-auto'>
          <div className="bg-base-2 w-full lg:w-[480px] max-w-[480px] backdrop-blur-sm rounded-3xl p-6 md:p-8 border-white/10">
            <div className="space-y-4 md:space-y-6">
              {/* Form Başlığı */}
              <div className="text-center">
                <h3 className="text-xl md:text-2xl font-bold text-white mb-2">Ücretsiz Üye Ol</h3>
              </div>

              {/* Email Form */}
              <form onSubmit={handleFormSubmit} className="space-y-3 md:space-y-4">
                {/* Ad Soyad Input - Yan Yana */}
                <div className="grid grid-cols-2 gap-3">
                  <Input
                    type="text"
                    name="firstName"
                    placeholder="Ad"
                    value={firstName}
                    onChange={(e) => setFirstName(e.target.value)}
                    required
                  />
                  <Input
                    type="text"
                    name="lastName"
                    placeholder="Soyad"
                    value={lastName}
                    onChange={(e) => setLastName(e.target.value)}
                    required
                  />
                </div>

                {/* Password Input - Ad veya soyad girildiğinde göster */}
                <div 
                  className={`transition-all duration-300 ease-in-out overflow-hidden ${
                    (firstName.trim() || lastName.trim()) ? 'max-h-40 opacity-100' : 'max-h-0 opacity-0'
                  }`}
                >
                  <Input
                    type="password"
                    name="password"
                    placeholder="Güçlü bir şifre oluşturun"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    required={(firstName.trim() !== '' || lastName.trim() !== '')}
                    showPasswordToggle
                    hint="En az 8 karakter"
                  />
                </div>

                <Button 
                  type="submit"
                  variant="primary" 
                  size="md"
                  className="w-full"
                >
                  Hemen Kazanmaya Başla
                </Button>
              </form>

              {/* Divider */}
              <div className="relative">
                <div className="absolute inset-0 flex items-center">
                  <div className="w-full border-t border-white/20"></div>
                </div>
                <div className="relative flex justify-center text-sm">
                  <span className="px-2 bg-base-2 text-gray-400">ya da</span>
                </div>
              </div>

              {/* Social Login Buttons */}
              <div className="grid grid-cols-2 gap-2 md:gap-3">
                {/* Google - Tam genişlik */}
                <Button
                  onClick={handleGoogleSignup}
                  variant="outline"
                  size="md"
                  fullWidth
                  className="col-span-2"
                >
                  <svg className="w-4 h-4 md:w-5 md:h-5" viewBox="0 0 24 24">
                    <path fill="#4285F4" d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z"/>
                    <path fill="#34A853" d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z"/>
                    <path fill="#FBBC05" d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z"/>
                    <path fill="#EA4335" d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z"/>
                  </svg>
                  <span>Google</span>
                </Button>

                {/* Facebook - Yarı genişlik */}
                <Button
                  onClick={handleFacebookSignup}
                  variant="outline"
                  size="md"
                  fullWidth
                >
                  <Iconify icon="logos:facebook" size={18} />
                  <span>Facebook</span>
                </Button>

                {/* Apple - Yarı genişlik */}
                <Button
                  onClick={handleAppleSignup}
                  variant="outline"
                  size="md"
                  fullWidth
                >
                  <svg className="w-4 h-4 md:w-5 md:h-5" fill="currentColor" viewBox="0 0 24 24">
                    <path d="M18.71 19.5c-.83 1.24-1.71 2.45-3.05 2.47-1.34.03-1.77-.79-3.29-.79-1.53 0-2 .77-3.27.82-1.31.05-2.3-1.32-3.14-2.53C4.25 17 2.94 12.45 4.7 9.39c.87-1.52 2.43-2.48 4.12-2.51 1.28-.02 2.5.87 3.29.87.78 0 2.26-1.07 3.81-.91.65.03 2.47.26 3.64 1.98-.09.06-2.17 1.28-2.15 3.81.03 3.02 2.65 4.03 2.68 4.04-.03.07-.42 1.44-1.38 2.83M13 3.5c.73-.83 1.94-1.46 2.94-1.5.13 1.17-.34 2.35-1.04 3.19-.69.85-1.83 1.51-2.95 1.42-.15-1.15.41-2.35 1.05-3.11z"/>
                  </svg>
                  <span>Apple</span>
                </Button>
              </div>

              {/* Son Kayıtlar */}
              <div className="text-center text-sm text-gray-400">
                <span>{recentSignups}</span>
              </div>
            </div>
          </div>
          </div>
        </div>
      </div>
    </section>
  )
}

export { SignupSection }
export default SignupSection