'use client'

import Link from "next/link"
import { usePathname } from "next/navigation"
import { Iconify } from "@/components/ui"

interface TabBarProps {
  onMenuToggle: () => void
}

interface TabItem {
  id: string
  label: string
  icon: string
  activeIcon: string
  href: string
}

export default function TabBar({ onMenuToggle }: TabBarProps) {
  const pathname = usePathname()

  const tabs: TabItem[] = [
    {
      id: 'home',
      label: 'Ana Sayfa',
      icon: 'solar:home-2-linear',
      activeIcon: 'solar:home-2-bold',
      href: '/home'
    },
    {
      id: 'tasks',
      label: 'Görevlerim',
      icon: 'solar:clipboard-list-linear',
      activeIcon: 'solar:clipboard-list-bold',
      href: '/my-duties'
    },
    {
      id: 'wallet',
      label: 'Bakiye Çek',
      icon: 'solar:wallet-linear',
      activeIcon: 'solar:wallet-bold',
      href: '/wallet'
    },
    {
      id: 'menu',
      label: 'Menü',
      icon: 'solar:hamburger-menu-linear',
      activeIcon: 'solar:hamburger-menu-bold',
      href: '#'
    }
  ]

  const handleTabClick = (tab: TabItem, e: React.MouseEvent) => {
    if (tab.id === 'menu') {
      e.preventDefault()
      onMenuToggle()
    }
  }

  const isActive = (href: string) => {
    if (href === '/home') {
      return pathname === '/home'
    }
    if (href === '#') {
      return false
    }
    return pathname.startsWith(href)
  }

  return (
    <div className="md:hidden fixed -bottom-6 pb-6 left-0 right-0 z-50 bg-base-1/85 border-t border-white/10 backdrop-blur-lg safe-area-bottom">
      <div className="grid grid-cols-4 h-16">
        {tabs.map((tab) => {
          const active = tab.id !== 'menu' && isActive(tab.href)
          
          if (tab.id === 'menu') {
            return (
              <button
                key={tab.id}
                onClick={(e) => handleTabClick(tab, e)}
                className="flex flex-col items-center justify-center gap-1 transition-all active:scale-95 active:bg-white/5"
              >
                <Iconify 
                  icon={tab.icon} 
                  className="text-2xl text-gray-400"
                />
                <span className="text-[10px] text-gray-400 font-medium">
                  {tab.label}
                </span>
              </button>
            )
          }

          return (
            <Link
              key={tab.id}
              href={tab.href}
              className="flex flex-col items-center justify-center gap-1 transition-all active:scale-95 active:bg-white/5 relative"
            >
              {active && (
                <div className="absolute top-0 left-1/2 -translate-x-1/2 w-12 h-1 bg-primary rounded-b-full" />
              )}
              <Iconify 
                icon={active ? tab.activeIcon : tab.icon} 
                className={`text-2xl transition-colors ${active ? 'text-primary' : 'text-gray-400'}`}
              />
              <span className={`text-[10px] font-medium transition-colors ${active ? 'text-primary' : 'text-gray-400'}`}>
                {tab.label}
              </span>
            </Link>
          )
        })}
      </div>
    </div>
  )
}
