'use client'

import React, { useState, useRef, useEffect } from 'react'
import { cn } from '@/lib/utils'
import { Icon, Button, ToggleSwitch } from '../ui'

import type { IconName } from '@/lib/icons'
import Image from 'next/image'

interface UserDropdownProps {
  className?: string
  user?: {
    name: string
    username: string
    balance: string
    avatar?: string
  }
}

interface MenuItem {
  icon: IconName
  label: string
  href: string
}

const UserDropdown: React.FC<UserDropdownProps> = ({ 
  className,
  user = {
    name: 'Arda',
    username: '@ardakilic',
    balance: '12,331₺'
  }
}) => {
  const [isOpen, setIsOpen] = useState(false)
  const [showBalance, setShowBalance] = useState(true)
  const dropdownRef = useRef<HTMLDivElement>(null)

  const menuItems: MenuItem[] = [
    { icon: 'fa6-solid:list', label: 'Ana Sayfa', href: '/' },
    { icon: 'heroicons:newspaper', label: 'Görevlerim', href: '/my-duties' },
    { icon: 'mdi:user-plus', label: 'Referanslarım', href: '/references' },
    { icon: 'weui:setting-filled', label: 'Ayarlar', href: '/settings' },
  ]

  const toggleDropdown = () => setIsOpen(!isOpen)

  // Dropdown dışına tıklandığında kapat
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false)
      }
    }

    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside)
    }

    return () => {
      document.removeEventListener('mousedown', handleClickOutside)
    }
  }, [isOpen])

  return (
    <div ref={dropdownRef} className={cn('relative', className)}>
      {/* Trigger Button */}
      <button
        onClick={toggleDropdown}
        className={cn(
          'flex items-center justify-start gap-2 p-2 w-32 rounded-lg transition-all duration-200',
          'hover:bg-base-2',
          isOpen && 'bg-base-2'
        )}
      >
        {/* Avatar */}
        <Image
          src={user.avatar || '/assets/images/pp1.png'}
          alt="User Avatar"
          width={32}
          height={32}
          className="w-8 h-8 rounded-full object-cover"
        />
        
        {/* User Info */}
        <div className="flex flex-col text-left">
          <span className="text-white text-sm font-medium">{user.name}</span>
          <div className="flex items-center gap-1">
            <div className="w-2 h-2 bg-green-500 rounded-full"></div>
            <span className="text-xs text-gray-400">Onaylı</span>
          </div>
        </div>

        {/* Dropdown Arrow - Sağa bakan ok */}
        <svg 
          className={cn(
            'w-4 h-4 text-gray-400 rotate-90  transition-transform duration-500 ',
            isOpen && '-rotate-90' // Açıkken yukarı (sağdan yukarı)
          )}
          fill="none"
          stroke="currentColor"
          viewBox="0 0 24 24"
        >
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
        </svg>
      </button>

      {/* Dropdown Menu */}
      {isOpen && (
        <>
          {/* Backdrop */}
            <div 
              className="fixed inset-0 z-40 h-screen"
              onClick={() => setIsOpen(false)}
            />
            
          {/* Menu */}
          <div className="absolute top-full right-0 mt-2 w-80 bg-base-2 rounded-xl border border-white/10 shadow-xl z-50 overflow-hidden animate-in fade-in slide-in-from-top-2 duration-200">
            {/* User Header */}
            <div className="relative p-4 border-b border-white/10 rounded-t-xl rounded-b-none bg-base-2">
              <div className="flex relative z-10 items-start justify-start gap-3">
                {/* Avatar */}
                <Image
                src={user.avatar || '/assets/images/pp1.png'}
                alt="User Avatar"
                width={64}
                height={64}
                className="w-12 h-12 border border-primary/20 rounded-full object-cover"
                />
                <div className="flex  flex-col items-start">
                  <h3 className="text-white font-medium">{user.name}</h3>
                  <p className="text-gray-400 text-sm">{user.username}</p>
                </div>
              </div>
              
              {/* Balance Section */}
              <div className="relative z-10 mt-3 flex  items-start justify-between">
                <div className='flex flex-col items-start'>
                  <p className="text-white/70 text-sm">Mevcut Bakiye</p>
                  <p className="text-green-400 font-bold text-lg">
                    {showBalance ? user.balance : '****₺'}
                  </p>
                </div>
                <Button  variant="outline" size="sm">
                  Bakiyeyi Çek
                </Button>
              </div>
              {/* Background Light Effect */}
              <span className='z-0 absolute rounded-full w-36 h-36 left-1/2 -top-1/2 -translate-x-1/2  blur-2xl  bg-primary'  />
              
            </div>

            {/* Menu Items */}
            <div className="p-2">
              {menuItems.map((item, index) => (
                <a
                  key={index}
                  href={item.href}
                  className={cn(
                    'flex items-center gap-3 px-3 py-2.5 rounded-lg',
                    'text-gray-300 hover:text-white hover:bg-base-3',
                    'transition-all duration-200 group'
                  )}
                  onClick={() => setIsOpen(false)}
                >
                    
                  <Icon 
                    name={item.icon} 
                    size="sm" 
                    className="text-gray-400  group-hover:text-white" 
                  />
                  <span className="text-sm">{item.label}</span>
                </a>
              ))}
               <button className="flex items-center gap-3 px-3 py-2.5 rounded-lg w-full text-left text-red-600 hover:text-red-500 hover:bg-red-700/10 transition-all duration-200 group">
                <svg 
                  className="w-4 h-4 text-red-600 group-hover:text-red-500"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1" />
                </svg>
                <span className="text-sm">Çıkış Yap</span>
              </button>

            </div>

            {/* Bottom Actions */}
            <div className="p-2 border-t border-base-3">
             
              {/* Toggle Switches */}
              <div className="mt-2 space-y-2">
                <div className="flex items-center justify-between px-3 py-2">
                  <span className="text-sm text-gray-300">TL&apos;yi Göster</span>
                  <ToggleSwitch 
                    defaultChecked={showBalance}
                    variant="primary"
                    size="md"
                    onChange={(checked) => setShowBalance(checked)}
                    labelPosition='left'
                    
                  />
                </div>
                
                <div className="flex items-center justify-between px-3 py-2">
                  <span className="text-sm text-gray-300">Canlı İstatistikler</span>
                  <ToggleSwitch 
                    defaultChecked={false}
                    variant="primary"
                    size="md"
                    onChange={
                        
                        (a) => {
                            if (a === true) {
                                console.log('Canlı İstatistikler: Açık')
                            } else {
                                console.log('Canlı İstatistikler: Kapalı')
                            }
                        }
                    }
                      labelPosition='right'
                  />
                </div>
              </div>            </div>
          </div>
        </>
      )}
    </div>
  )
}

export default UserDropdown