'use client'

import React, { useState } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { Modal, Button, Input, Iconify } from '../ui'

interface ForgotPasswordModalProps {
  isOpen: boolean
  onClose: () => void
  onBackToLogin: () => void
}

export function ForgotPasswordModal({ isOpen, onClose, onBackToLogin }: ForgotPasswordModalProps) {
  const [step, setStep] = useState<'email' | 'code' | 'newPassword' | 'success'>('email')
  const [formData, setFormData] = useState({
    email: '',
    code: '',
    newPassword: '',
    confirmPassword: ''
  })
  const [isLoading, setIsLoading] = useState(false)
  const [errors, setErrors] = useState<Record<string, string>>({})

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target
    setFormData(prev => ({
      ...prev,
      [name]: value
    }))
    
    // Clear error when user starts typing
    if (errors[name]) {
      setErrors(prev => ({ ...prev, [name]: '' }))
    }
  }

  const handleEmailSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)
    setErrors({})

    // Validation
    const newErrors: Record<string, string> = {}
    
    if (!formData.email) {
      newErrors.email = 'E-posta adresi gerekli'
    } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
      newErrors.email = 'Geçerli bir e-posta adresi girin'
    }

    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors)
      setIsLoading(false)
      return
    }

    try {
      // Simulate API call
      console.log('Sending reset code to:', formData.email)
      await new Promise(resolve => setTimeout(resolve, 1000))
      
      // Move to code verification step
      setStep('code')
      
    } catch (error) {
      setErrors({ general: 'Kod gönderilirken bir hata oluştu' })
    } finally {
      setIsLoading(false)
    }
  }

  const handleCodeSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)
    setErrors({})

    // Validation
    const newErrors: Record<string, string> = {}
    
    if (!formData.code) {
      newErrors.code = 'Doğrulama kodu gerekli'
    } else if (formData.code.length !== 6) {
      newErrors.code = 'Kod 6 haneli olmalı'
    }

    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors)
      setIsLoading(false)
      return
    }

    try {
      // Simulate API call
      console.log('Verifying code:', formData.code)
      await new Promise(resolve => setTimeout(resolve, 1000))
      
      // Move to new password step
      setStep('newPassword')
      
    } catch (error) {
      setErrors({ general: 'Kod doğrulanamadı' })
    } finally {
      setIsLoading(false)
    }
  }

  const handlePasswordSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)
    setErrors({})

    // Validation
    const newErrors: Record<string, string> = {}
    
    if (!formData.newPassword) {
      newErrors.newPassword = 'Yeni şifre gerekli'
    } else if (formData.newPassword.length < 6) {
      newErrors.newPassword = 'Şifre en az 6 karakter olmalı'
    }
    
    if (!formData.confirmPassword) {
      newErrors.confirmPassword = 'Şifre tekrarı gerekli'
    } else if (formData.newPassword !== formData.confirmPassword) {
      newErrors.confirmPassword = 'Şifreler eşleşmiyor'
    }

    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors)
      setIsLoading(false)
      return
    }

    try {
      // Simulate API call
      console.log('Resetting password')
      await new Promise(resolve => setTimeout(resolve, 1000))
      
      // Move to success step
      setStep('success')
      
    } catch (error) {
      setErrors({ general: 'Şifre sıfırlanırken bir hata oluştu' })
    } finally {
      setIsLoading(false)
    }
  }

  const handleResendCode = async () => {
    setIsLoading(true)
    try {
      console.log('Resending code to:', formData.email)
      await new Promise(resolve => setTimeout(resolve, 1000))
      // Show success message (you can add a toast notification here)
    } catch (error) {
      setErrors({ general: 'Kod tekrar gönderilemedi' })
    } finally {
      setIsLoading(false)
    }
  }

  const resetModal = () => {
    setStep('email')
    setFormData({
      email: '',
      code: '',
      newPassword: '',
      confirmPassword: ''
    })
    setErrors({})
  }

  const handleClose = () => {
    resetModal()
    onClose()
  }

  const handleBackToLoginClick = () => {
    resetModal()
    onBackToLogin()
  }

  return (
    <Modal 
      isOpen={isOpen} 
      onClose={handleClose}
      size="md"
    >
      <div className="space-y-4 md:space-y-8">
        <AnimatePresence mode="wait">
        {/* Email Step */}
        {step === 'email' && (
          <motion.div
            key="email"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            exit={{ opacity: 0, x: 20 }}
            transition={{ duration: 0.3 }}
          >
            <div>
              <h2 className="text-xl md:text-2xl font-semibold text-white text-start mb-2">
                Şifremi Unuttum
              </h2>
              <p className="text-sm text-gray-400">
                E-posta adresinize şifre sıfırlama kodu göndereceğiz.
              </p>
            </div>

            <form onSubmit={handleEmailSubmit} className="space-y-8  md:space-y-10">
              {errors.general && (
                <div className="p-3 bg-red-500/10 border border-red-500/20 rounded-lg">
                  <p className="text-sm text-red-400">{errors.general}</p>
                </div>
              )}

              <Input
                name="email"
                type="email"
                label="E-posta Adresi"
                placeholder="ornek@email.com"
                value={formData.email}
                onChange={handleInputChange}
                error={errors.email}
                required
              />

              <Button
                type="submit"
                variant="primary"
                size="lg"
                className="w-full"
                disabled={isLoading}
              >
                {isLoading ? (
                  <div className="flex items-center gap-2">
                    <div className="w-4 h-4 border-2 border-white/30 border-t-white rounded-full animate-spin" />
                    <span className="text-sm md:text-base">Kod Gönderiliyor...</span>
                  </div>
                ) : (
                  <span className="text-sm md:text-base">Kod Gönder</span>
                )}
              </Button>

              <div className="text-center pt-3 md:pt-4 border-t border-base-3">
                <button
                  type="button"
                  onClick={handleBackToLoginClick}
                  className="text-sm text-gray-400 hover:text-white transition-colors flex items-center gap-2 mx-auto"
                >
                  <Iconify icon="mdi:arrow-left" size={16} />
                  Giriş sayfasına dön
                </button>
              </div>
            </form>
          </motion.div>
        )}

        {/* Code Verification Step */}
        {step === 'code' && (
          <motion.div
            key="code"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            exit={{ opacity: 0, x: 20 }}
            transition={{ duration: 0.3 }}
          >
            <div>
              <h2 className="text-xl md:text-2xl font-semibold text-white text-start mb-2">
                Kodu Doğrula
              </h2>
              <p className="text-sm text-gray-400">
                <span className="text-primary">{formData.email}</span> adresine gönderilen 6 haneli kodu girin.
              </p>
            </div>

            <form onSubmit={handleCodeSubmit} className="space-y-3 md:space-y-4">
              {errors.general && (
                <div className="p-3 bg-red-500/10 border border-red-500/20 rounded-lg">
                  <p className="text-sm text-red-400">{errors.general}</p>
                </div>
              )}

              <Input
                name="code"
                type="text"
                label="Doğrulama Kodu"
                placeholder="123456"
                value={formData.code}
                onChange={handleInputChange}
                error={errors.code}
                required
                maxLength={6}
              />

              <Button
                type="submit"
                variant="primary"
                size="lg"
                className="w-full"
                disabled={isLoading}
              >
                {isLoading ? (
                  <div className="flex items-center gap-2">
                    <div className="w-4 h-4 border-2 border-white/30 border-t-white rounded-full animate-spin" />
                    <span className="text-sm md:text-base">Doğrulanıyor...</span>
                  </div>
                ) : (
                  <span className="text-sm md:text-base">Doğrula</span>
                )}
              </Button>

              <div className="text-center">
                <p className="text-sm text-gray-400">
                  Kod gelmedi mi?{' '}
                  <button
                    type="button"
                    onClick={handleResendCode}
                    disabled={isLoading}
                    className="text-primary hover:text-primary/80 font-medium transition-colors disabled:opacity-50"
                  >
                    Tekrar gönder
                  </button>
                </p>
              </div>

              <div className="text-center pt-3 md:pt-4 border-t border-base-3">
                <button
                  type="button"
                  onClick={() => setStep('email')}
                  className="text-sm text-gray-400 hover:text-white transition-colors flex items-center gap-2 mx-auto"
                >
                  <Iconify icon="mdi:arrow-left" size={16} />
                  E-posta değiştir
                </button>
              </div>
            </form>
          </motion.div>
        )}

        {/* New Password Step */}
        {step === 'newPassword' && (
          <motion.div
            key="newPassword"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            exit={{ opacity: 0, x: 20 }}
            transition={{ duration: 0.3 }}
          >
            <div>
              <h2 className="text-xl md:text-2xl font-semibold text-white text-start mb-2">
                Yeni Şifre Oluştur
              </h2>
              <p className="text-sm text-gray-400">
                Hesabınız için yeni bir şifre belirleyin.
              </p>
            </div>

            <form onSubmit={handlePasswordSubmit} className="space-y-3 md:space-y-4">
              {errors.general && (
                <div className="p-3 bg-red-500/10 border border-red-500/20 rounded-lg">
                  <p className="text-sm text-red-400">{errors.general}</p>
                </div>
              )}

              <Input
                name="newPassword"
                type="password"
                label="Yeni Şifre"
                placeholder="En az 6 karakter"
                value={formData.newPassword}
                onChange={handleInputChange}
                error={errors.newPassword}
                required
                showPasswordToggle
              />

              <Input
                name="confirmPassword"
                type="password"
                label="Şifre Tekrar"
                placeholder="Şifrenizi tekrar girin"
                value={formData.confirmPassword}
                onChange={handleInputChange}
                error={errors.confirmPassword}
                required
                showPasswordToggle
              />

              <Button
                type="submit"
                variant="primary"
                size="lg"
                className="w-full"
                disabled={isLoading}
              >
                {isLoading ? (
                  <div className="flex items-center gap-2">
                    <div className="w-4 h-4 border-2 border-white/30 border-t-white rounded-full animate-spin" />
                    <span className="text-sm md:text-base">Şifre Sıfırlanıyor...</span>
                  </div>
                ) : (
                  <span className="text-sm md:text-base">Şifreyi Değiştir</span>
                )}
              </Button>
            </form>
          </motion.div>
        )}

        {/* Success Step */}
        {step === 'success' && (
          <motion.div
            key="success"
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            exit={{ opacity: 0, scale: 0.9 }}
            transition={{ duration: 0.3 }}
          >
            <div className="text-center py-4 md:py-8">
              <motion.div 
                className="w-16 h-16 md:w-20 md:h-20 mx-auto mb-4 bg-green-500/20 rounded-full flex items-center justify-center"
                initial={{ scale: 0, rotate: -180 }}
                animate={{ scale: 1, rotate: 0 }}
                transition={{ 
                  delay: 0.2,
                  type: "spring" as const,
                  stiffness: 200,
                  damping: 15
                }}
              >
                <Iconify icon="mdi:check-circle" className="text-green-500" size={40} />
              </motion.div>
              
              <h2 className="text-xl md:text-2xl font-semibold text-white mb-2">
                Şifreniz Değiştirildi!
              </h2>
              <p className="text-sm text-gray-400 mb-6">
                Şifreniz başarıyla değiştirildi. Artık yeni şifrenizle giriş yapabilirsiniz.
              </p>

              <Button
                variant="primary"
                size="lg"
                className="w-full"
                onClick={handleBackToLoginClick}
              >
                Giriş Yap
              </Button>
            </div>
          </motion.div>
        )}
        </AnimatePresence>
      </div>
    </Modal>
  )
}
