'use client'

import React, { useState, useEffect } from 'react'
import { LiveSupport, type ChatMessage } from '../../app/(dashboard)/_components/LiveSupport'
import { Iconify } from '../ui'
import { getLiveChatMessages, generateMessageId, getFormattedTime } from '@/data'

export interface LiveSupportWidgetProps {
  autoOpen?: boolean
  welcomeMessage?: string
}

const LiveSupportWidget: React.FC<LiveSupportWidgetProps> = ({
  autoOpen = false,
  welcomeMessage = "Merhaba! Size nasıl yardımcı olabiliriz?"
}) => {
  const [isOpen, setIsOpen] = useState(autoOpen)
  const [messages, setMessages] = useState<ChatMessage[]>([])
  const [hasNewMessage, setHasNewMessage] = useState(false)

  useEffect(() => {
    const liveChatMessages = getLiveChatMessages()
    const formattedMessages = liveChatMessages.map(msg => ({
      ...msg,
      avatar: msg.avatar || '/assets/images/pp1.png'
    }))
    setMessages(formattedMessages)
  }, [])

  useEffect(() => {
    if (!isOpen && messages.length > 0) {
      setHasNewMessage(true)
    } else {
      setHasNewMessage(false)
    }
  }, [isOpen, messages.length])

  const handleSendMessage = (message: string) => {
    const newMsg: ChatMessage = {
      id: generateMessageId(),
      sender: 'Siz',
      avatar: '/assets/images/pp1.png',
      message,
      time: getFormattedTime(),
      isOnline: true
    }
    setMessages([...messages, newMsg])
    
    setTimeout(() => {
      const response: ChatMessage = {
        id: generateMessageId(),
        sender: 'Destek Ekibi',
        avatar: '/assets/images/pp1.png',
        message: 'Mesajınız alındı. En kısa sürede size yardımcı olacağız!',
        time: getFormattedTime(),
        isOnline: true
      }
      setMessages(prev => [...prev, response])
    }, 2000)
  }

  const toggleChat = () => {
    setIsOpen(!isOpen)
    if (!isOpen) {
      setHasNewMessage(false)
    }
  }

  return (
    <>
      <div className="hidden lg:block">
        <LiveSupport
          isOpen={isOpen}
          messages={messages}
          onSendMessage={handleSendMessage}
          onClose={() => setIsOpen(false)}
        />
      </div>

      {!isOpen && (
        <button
          onClick={toggleChat}
          className="hidden lg:flex fixed bottom-4 right-4 w-14 h-14 bg-primary hover:bg-primary/80 rounded-full items-center justify-center shadow-lg transition-all duration-300 hover:scale-110 z-40 group"
        >
          <Iconify 
            icon="solar:chat-round-dots-bold" 
            size="lg" 
            className="text-white group-hover:scale-110 transition-transform" 
          />
          
          {hasNewMessage && (
            <div className="absolute -top-1 -right-1 w-6 h-6 bg-red-500 rounded-full flex items-center justify-center animate-pulse">
              <span className="text-white text-xs font-bold">!</span>
            </div>
          )}
        </button>
      )}
    </>
  )
}

export { LiveSupportWidget }
export default LiveSupportWidget
