'use client'

import { useState, useEffect } from "react"
import { Iconify } from "@/components/ui"
import Image from "next/image"
import { getLiveChatMessages, generateMessageId, getFormattedTime, type LiveChatMessage } from '@/data'

interface MobileLiveChatProps {
  isOpen: boolean
  onClose: () => void
}

export default function MobileLiveChat({ isOpen, onClose }: MobileLiveChatProps) {
  const [messages, setMessages] = useState<LiveChatMessage[]>([])
  const [newMessage, setNewMessage] = useState('')
  const [isTyping, setIsTyping] = useState(false)

  // Mesajları yükle
  useEffect(() => {
    setMessages(getLiveChatMessages())
  }, [])

  // Modal açıkken arka planın scroll olmasını engelle
  useEffect(() => {
    if (isOpen) {
      // Scroll pozisyonunu kaydet
      const scrollY = window.scrollY
      document.body.style.position = 'fixed'
      document.body.style.top = `-${scrollY}px`
      document.body.style.width = '100%'
    } else {
      // Scroll pozisyonunu geri yükle
      const scrollY = document.body.style.top
      document.body.style.position = ''
      document.body.style.top = ''
      document.body.style.width = ''
      if (scrollY) {
        window.scrollTo(0, parseInt(scrollY || '0') * -1)
      }
    }

    return () => {
      document.body.style.position = ''
      document.body.style.top = ''
      document.body.style.width = ''
    }
  }, [isOpen])

  const handleSendMessage = () => {
    if (newMessage.trim()) {
      // Kullanıcı mesajını ekle
      const userMsg: LiveChatMessage = {
        id: generateMessageId(),
        sender: 'Siz',
        message: newMessage,
        time: getFormattedTime(),
        isUser: true
      }
      
      setMessages(prev => [...prev, userMsg])
      setNewMessage('')
      
      // Yazıyor göstergesini göster
      setIsTyping(true)
      
      // 2 saniye sonra otomatik cevap
      setTimeout(() => {
        setIsTyping(false)
        const supportMsg: LiveChatMessage = {
          id: generateMessageId(),
          sender: 'Destek Ekibi',
          avatar: '/assets/images/pp1.png',
          message: 'Mesajınızı aldık. En kısa sürede size dönüş yapacağız. 🙏',
          time: getFormattedTime(),
          isOnline: true,
          isUser: false
        }
        setMessages(prev => [...prev, supportMsg])
      }, 2000)
    }
  }

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault()
      handleSendMessage()
    }
  }

  if (!isOpen) return null

  return (
    <>
      {/* Backdrop */}
      <div 
        className="md:hidden fixed inset-0 bg-black/60 backdrop-blur-sm z-50 animate-in fade-in duration-200"
        onClick={onClose}
      />
      
      {/* Chat Panel */}
      <div className="md:hidden pb-3 fixed inset-x-0 -bottom-2 z-50 bg-base-1 rounded-t-3xl animate-in slide-in-from-bottom duration-300 h-[85vh] flex flex-col">
        {/* Header */}
        <div className="flex items-center justify-between px-4 py-4 border-b border-white/10">
          <div className="flex items-center gap-3">
            <div className="relative">
             <Iconify icon="gridicons:chat" size="lg" />
              </div>
            <div>
              <h3 className="text-white font-semibold text-sm">Canlı Sohbet</h3>
             
            </div>
          </div>
          
          <button
            onClick={onClose}
            className="p-2 hover:bg-white/5 rounded-full transition-colors"
          >
            <Iconify icon="mdi:close" className="text-gray-400 text-xl" />
          </button>
        </div>

        {/* Messages */}
        <div className="flex-1 overflow-y-auto custom-scrollbar p-4 space-y-4">
          {messages.map((msg) => (
            <div 
              key={msg.id}
              className={`flex items-end  gap-2 ${msg.isUser ? 'flex-row-reverse' : ''}`}
            >
              {!msg.isUser && msg.avatar && (
                <Image
                  src={msg.avatar}
                  alt={msg.sender}
                  width={32}
                  height={32}
                  className="w-8 h-8 rounded-full object-cover flex-shrink-0"
                />
              )}
              
              <div className={`flex flex-col ${msg.isUser ? 'items-end' : 'items-start'} flex-1 min-w-0`}>
                {!msg.isUser && (
                  <span className="text-xs text-white/60 mb-1">{msg.sender}</span>
                )}
                <div className={`relative rounded-2xl px-4 py-2.5  ${
                  msg.isUser
                    ? 'bg-primary/20 border border-primary/30 rounded-br-none'
                    : 'bg-base-3 rounded-bl-none'
                }`}>
                  {!msg.isUser && (
                    <span className='absolute -left-2 bottom-0'>
                      <svg width="20" height="20" viewBox="0 0 27 28" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M7.09989 21.8499C11.6643 16.8499 10.65 0.849945 10.65 0.849945L26.8789 20.3499C20.7931 25.3499 11.6643 28.8499 -0.000267029 27.3499C-0.000267029 27.3499 2.5355 26.8499 7.09989 21.8499Z" fill="#211D2F"/>
                      </svg>
                    </span>
                  )}
                  {msg.isUser && (
                    <span className='absolute -right-2 bottom-0 scale-x-[-1]'>
                      <svg width="20" height="20" viewBox="0 0 27 28" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M7.09989 21.8499C11.6643 16.8499 10.65 0.849945 10.65 0.849945L26.8789 20.3499C20.7931 25.3499 11.6643 28.8499 -0.000267029 27.3499C-0.000267029 27.3499 2.5355 26.8499 7.09989 21.8499Z" fill="rgba(139, 92, 246, 0.2)"/>
                      </svg>
                    </span>
                  )}
                  <p className={`text-sm leading-relaxed ${msg.isUser ? 'text-white' : 'text-gray-300'}`}>
                    {msg.message}
                    <span className={`text-xs absolute right-2 bottom-2 ${msg.isUser ? 'text-gray-400' : 'text-gray-500'}`}>
                  {msg.time}
                </span>
                  </p>
                </div>
                
              </div>
            </div>
          ))}
          
          {/* Typing indicator */}
          {isTyping && (
            <div className="flex items-start gap-2">
              <Image
                src="/assets/images/pp1.png"
                alt="Support"
                width={32}
                height={32}
                className="w-8 h-8 rounded-full object-cover flex-shrink-0"
              />
              <div className="bg-base-3 rounded-2xl rounded-bl-none px-4 py-3">
                <div className="flex gap-1">
                  <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0ms' }} />
                  <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '150ms' }} />
                  <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '300ms' }} />
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Input */}
        <div className="p-4 border-t border-white/10 bg-base-1">
          <div className="flex items-center gap-2">
            <button className="p-2 flex items-center justify-center hover:bg-white/5 rounded-full transition-colors">
              <Iconify icon="mdi:emoticon-happy-outline" className="text-gray-400 text-xl" />
            </button>
            
            <input
              type="text"
              value={newMessage}
              onChange={(e) => setNewMessage(e.target.value)}
              onKeyPress={handleKeyPress}
              placeholder="Mesajınızı yazın..."
              className="flex-1 bg-base-3 text-white px-4 py-2.5 rounded-full text-sm focus:outline-none focus:ring-2 focus:ring-primary/50 transition-all"
            />
            
            <button
              onClick={handleSendMessage}
              disabled={!newMessage.trim()}
              className="p-2.5 flex items-center justify-center bg-primary hover:bg-primary/90 disabled:opacity-50 disabled:cursor-not-allowed rounded-full transition-all active:scale-95"
            >
              <Iconify icon="mdi:send" className="text-white text-xl" />
            </button>
          </div>
        </div>
      </div>
    </>
  )
}
