'use client'

import React, { useState } from 'react'
import { useRouter } from 'next/navigation'
import { motion, AnimatePresence } from 'framer-motion'
import { Modal, Button, Input, Iconify, Checkbox } from '../ui'
import { StatCard } from '@/app/(dashboard)/_components'

interface RegisterModalProps {
  isOpen: boolean
  onClose: () => void
  onSwitchToLogin: () => void
}

export function RegisterModal({ isOpen, onClose, onSwitchToLogin }: RegisterModalProps) {
  const router = useRouter()
  const [currentStep, setCurrentStep] = useState(1)
  const [showSuccessModal, setShowSuccessModal] = useState(false)
  const [showReferralInput, setShowReferralInput] = useState(false)
  const [formData, setFormData] = useState({
    // Step 1
    firstName: '',
    lastName: '',
    password: '',
    acceptTerms: false,
    acceptMarketing: false,
    // Step 2
    username: '',
    email: '',
    phone: '',
    birthMonth: '',
    birthDay: '',
    birthYear: '',
    gender: '',
    country: '',
    city: '',
    referralCode: ''
  })
  const [isLoading, setIsLoading] = useState(false)
  const [errors, setErrors] = useState<Record<string, string>>({})

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target
    const checked = (e.target as HTMLInputElement).checked
    
    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? checked : value
    }))
    
    // Clear error when user starts typing
    if (errors[name]) {
      setErrors(prev => ({ ...prev, [name]: '' }))
    }
  }

  const validateStep1 = () => {
    const newErrors: Record<string, string> = {}
    
    if (!formData.firstName.trim()) {
      newErrors.firstName = 'Ad gerekli'
    }
    
    if (!formData.lastName.trim()) {
      newErrors.lastName = 'Soyad gerekli'
    }
    
    if (!formData.password) {
      newErrors.password = 'Şifre gerekli'
    } else if (formData.password.length < 8) {
      newErrors.password = 'Şifre en az 8 karakter olmalı'
    } else if (!/(?=.*[a-z])(?=.*[A-Z])/.test(formData.password)) {
      newErrors.password = 'Şifre en az bir büyük harf, bir küçük harf ve bir rakam içermeli'
    }
    
    if (!formData.acceptTerms) {
      newErrors.acceptTerms = 'Kullanım şartlarını kabul etmelisiniz'
    }

    return newErrors
  }

  const validateStep2 = () => {
    const newErrors: Record<string, string> = {}
    
    if (!formData.username.trim()) {
      newErrors.username = 'Kullanıcı adı gerekli'
    } else if (formData.username.length < 3) {
      newErrors.username = 'Kullanıcı adı en az 3 karakter olmalı'
    }
    
    if (!formData.email) {
      newErrors.email = 'E-posta adresi gerekli'
    } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
      newErrors.email = 'Geçerli bir e-posta adresi girin'
    }
    
    if (!formData.phone) {
      newErrors.phone = 'Telefon numarası gerekli'
    } else if (!/^\+?[\d\s\-\(\)]{10,}$/.test(formData.phone)) {
      newErrors.phone = 'Geçerli bir telefon numarası girin'
    }
    
    if (!formData.birthMonth || !formData.birthDay || !formData.birthYear) {
      newErrors.birthDate = 'Doğum tarihi gerekli'
    }
    
    if (!formData.gender) {
      newErrors.gender = 'Cinsiyet seçimi gerekli'
    }
    
    if (!formData.country) {
      newErrors.country = 'Ülke seçimi gerekli'
    }
    
    if (!formData.city) {
      newErrors.city = 'Şehir gerekli'
    }

    return newErrors
  }

  const handleNextStep = () => {
    const stepErrors = validateStep1()
    if (Object.keys(stepErrors).length > 0) {
      setErrors(stepErrors)
      return
    }
    
    setErrors({})
    setCurrentStep(2)
  }

  const handlePrevStep = () => {
    setCurrentStep(1)
    setErrors({})
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)
    setErrors({})

    const stepErrors = validateStep2()
    if (Object.keys(stepErrors).length > 0) {
      setErrors(stepErrors)
      setIsLoading(false)
      return
    }

    try {
      // Simulate API call
      console.log('Register attempt:', formData)
      await new Promise(resolve => setTimeout(resolve, 1500))
      
      // Success - show success modal
      setShowSuccessModal(true)
      
    } catch (error) {
      setErrors({ general: 'Kayıt olurken bir hata oluştu' })
    } finally {
      setIsLoading(false)
    }
  }

  const handleSocialRegister = (provider: 'google' | 'apple' | 'facebook') => {
    console.log(`${provider} ile kayıt olunuyor...`)
    // Show step 2 modal for additional info
    setCurrentStep(2)
  }

  const handleClose = () => {
    onClose()
    setCurrentStep(1)
    setErrors({})
    setShowSuccessModal(false)
    setShowReferralInput(false)
  }

  const handleSuccessModalClose = () => {
    setShowSuccessModal(false)
    handleClose()
    router.push('/home')
  }

  // Generate year options (18-80 years old)
  const currentYear = new Date().getFullYear()
  const yearOptions = Array.from({ length: 63 }, (_, i) => currentYear - 18 - i)
  
  // Generate day options
  const dayOptions = Array.from({ length: 31 }, (_, i) => i + 1)
  
  // Month options
  const monthOptions = [
    { value: '1', label: 'Ocak' },
    { value: '2', label: 'Şubat' },
    { value: '3', label: 'Mart' },
    { value: '4', label: 'Nisan' },
    { value: '5', label: 'Mayıs' },
    { value: '6', label: 'Haziran' },
    { value: '7', label: 'Temmuz' },
    { value: '8', label: 'Ağustos' },
    { value: '9', label: 'Eylül' },
    { value: '10', label: 'Ekim' },
    { value: '11', label: 'Kasım' },
    { value: '12', label: 'Aralık' }
  ]

  // Türkiye'deki 81 il
  const cityOptions = [
    'Adana', 'Adıyaman', 'Afyonkarahisar', 'Ağrı', 'Amasya', 'Ankara', 'Antalya', 'Artvin',
    'Aydın', 'Balıkesir', 'Bilecik', 'Bingöl', 'Bitlis', 'Bolu', 'Burdur', 'Bursa',
    'Çanakkale', 'Çankırı', 'Çorum', 'Denizli', 'Diyarbakır', 'Edirne', 'Elazığ', 'Erzincan',
    'Erzurum', 'Eskişehir', 'Gaziantep', 'Giresun', 'Gümüşhane', 'Hakkari', 'Hatay', 'Isparta',
    'Mersin', 'İstanbul', 'İzmir', 'Kars', 'Kastamonu', 'Kayseri', 'Kırklareli', 'Kırşehir',
    'Kocaeli', 'Konya', 'Kütahya', 'Malatya', 'Manisa', 'Kahramanmaraş', 'Mardin', 'Muğla',
    'Muş', 'Nevşehir', 'Niğde', 'Ordu', 'Rize', 'Sakarya', 'Samsun', 'Siirt',
    'Sinop', 'Sivas', 'Tekirdağ', 'Tokat', 'Trabzon', 'Tunceli', 'Şanlıurfa', 'Uşak',
    'Van', 'Yozgat', 'Zonguldak', 'Aksaray', 'Bayburt', 'Karaman', 'Kırıkkale', 'Batman',
    'Şırnak', 'Bartın', 'Ardahan', 'Iğdır', 'Yalova', 'Karabük', 'Kilis', 'Osmaniye', 'Düzce'
  ]

  const renderStep1 = () => (
    <motion.form 
      onSubmit={(e) => { e.preventDefault(); handleNextStep(); }} 
      className="space-y-4"
      initial={{ opacity: 0, x: -20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: 20 }}
      transition={{ duration: 0.3 }}
    >
      {errors.general && (
        <div className="p-3 bg-red-500/10 border border-red-500/20 rounded-lg">
          <p className="text-sm text-red-400">{errors.general}</p>
        </div>
      )}

      <div className="grid grid-cols-2 gap-3">
        <Input
          name="firstName"
          type="text"
          label="Ad"
          placeholder="Adınız"
          value={formData.firstName}
          onChange={handleInputChange}
          error={errors.firstName}
          required
        //  leftIcon={<Iconify icon="material-symbols:person-outline" size={20} />}
        />

        <Input
          name="lastName"
          type="text"
          label="Soyad"
          placeholder="Soyadınız"
          value={formData.lastName}
          onChange={handleInputChange}
          error={errors.lastName}
          required
       //   leftIcon={<Iconify icon="material-symbols:person-outline" size={20} />}
        />
      </div>

      <Input
        name="password"
        type="password"
        label="Şifre"
        placeholder="Güçlü bir şifre oluşturun"
        value={formData.password}
        onChange={handleInputChange}
        error={errors.password}
        required
        showPasswordToggle
       // leftIcon={<Iconify icon="material-symbols:lock-outline" size={20} />}
        hint="En az 8 karakter, büyük/küçük harf ve rakam içermeli"
      />

      <div className="space-y-3">
        <Checkbox
          name="acceptTerms"
          checked={formData.acceptTerms}
          onChange={(checked) => setFormData(prev => ({ ...prev, acceptTerms: checked }))}
          required
          error={errors.acceptTerms}
          label={
            <span className="text-sm text-gray-300">
              <a href="/terms" className="text-primary hover:text-primary/80">
                Kullanım Şartları
              </a>
              {' '} ve {' '}
              <a href="/privacy" className="text-primary hover:text-primary/80">
                Gizlilik Politikası
              </a>
              &apos;nı okudum ve kabul ediyorum
            </span>
          }
          size="md"
          variant="primary"
        />

        <Checkbox
          name="acceptMarketing"
          checked={formData.acceptMarketing}
          onChange={(checked) => setFormData(prev => ({ ...prev, acceptMarketing: checked }))}
          label="Kampanya ve promosyon e-postalarını almak istiyorum"
          size="md"
          variant="primary"
          className="text-sm text-gray-300"
        />
      </div>

      <Button
        type="submit"
        variant="primary"
        size="md"
        className="w-full"
      >
        Devam Et
        <Iconify icon="material-symbols:arrow-forward" size={16} className="ml-2" />
      </Button>

      <div className="relative my-6">
        <div className="absolute inset-0 flex items-center">
          <div className="w-full border-t border-base-3" />
        </div>
        <div className="relative flex justify-center text-sm">
          <span className="px-4 bg-base-1 text-gray-400">veya</span>
        </div>
      </div>

      <div className="grid grid-cols-2 gap-3">
        {/* Google - Tam genişlik */}
        <Button
          type="button"
          variant="outline"
          onClick={() => handleSocialRegister('google')}
          size='md'
          fullWidth
          className="col-span-2"
        >
          <Iconify icon="logos:google-icon" size={18} />
          Google
        </Button>
        
        {/* Facebook - Yarı genişlik */}
        <Button
          type="button"
          variant="outline"
          onClick={() => handleSocialRegister('facebook')}
          size='md'
          fullWidth
        >
          <Iconify icon="logos:facebook" size={18} />
          Facebook
        </Button>

        {/* Apple - Yarı genişlik */}
        <Button
          type="button"
          variant="outline"
          onClick={() => handleSocialRegister('apple')}
          size='md'
          fullWidth
        >
          <Iconify icon="ic:baseline-apple" size={18} />
          Apple
        </Button>
      </div>

      <div className="text-center pt-4 border-t border-base-3">
        <p className="text-sm text-gray-400">
          Zaten hesabınız var mı?{' '}
          <button
            type="button"
            onClick={onSwitchToLogin}
            className="text-primary hover:text-primary/80 font-medium transition-colors"
          >
            Giriş yap
          </button>
        </p>
      </div>
    </motion.form>
    
  )

  const renderStep2 = () => (
    <motion.form 
      onSubmit={handleSubmit} 
      className="space-y-4"
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      transition={{ duration: 0.3 }}
    >
      {errors.general && (
        <div className="p-3 bg-red-500/10 border border-red-500/20 rounded-lg">
          <p className="text-sm text-red-400">{errors.general}</p>
        </div>
      )}

      <Input
        name="username"
        type="text"
        label="Kullanıcı Adı"
        placeholder="Benzersiz kullanıcı adınız"
        value={formData.username}
        onChange={handleInputChange}
        error={errors.username}
        required
       // leftIcon={<Iconify icon="material-symbols:alternate-email" size={20} />}
        hint="En az 3 karakter, sadece harf, rakam ve alt çizgi"
      />

      <Input
        name="email"
        type="email"
        label="E-posta Adresi"
        placeholder="ornek@email.com"
        value={formData.email}
        onChange={handleInputChange}
        error={errors.email}
        required
       // leftIcon={<Iconify icon="material-symbols:mail-outline" size={20} />}
      />

      <Input
        name="phone"
        type="tel"
        label="Telefon Numarası"
        placeholder="+90 5xx xxx xx xx"
        value={formData.phone}
        onChange={handleInputChange}
        error={errors.phone}
        required
       // leftIcon={<Iconify icon="material-symbols:phone" size={20} />}
      />

      {/* Birth Date */}
      <div className="space-y-2">
        <label className="block text-sm font-medium text-gray-200">
          Doğum Tarihi <span className="text-red-400">*</span>
        </label>
        <div className="grid grid-cols-3 gap-3">
          <select
            name="birthDay"
            value={formData.birthDay}
            onChange={handleInputChange}
            className="px-4 py-3  rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
            required
          >
            <option value="">Gün</option>
            {dayOptions.map(day => (
              <option key={day} value={day}>{day}</option>
            ))}
          </select>
          
          <select
            name="birthMonth"
            value={formData.birthMonth}
            onChange={handleInputChange}
            className="px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
            required
          >
            <option value="">Ay</option>
            {monthOptions.map(month => (
              <option key={month.value} value={month.value}>{month.label}</option>
            ))}
          </select>
          
          <select
            name="birthYear"
            value={formData.birthYear}
            onChange={handleInputChange}
            className="px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
            required
          >
            <option value="">Yıl</option>
            {yearOptions.map(year => (
              <option key={year} value={year}>{year}</option>
            ))}
          </select>
        </div>
        {errors.birthDate && (
          <p className="text-sm text-red-400">{errors.birthDate}</p>
        )}
      </div>

      {/* Gender */}
      <div className="space-y-2">
        <label className="block text-sm font-medium text-gray-200">
          Cinsiyet <span className="text-red-400">*</span>
        </label>
        <select
          name="gender"
          value={formData.gender}
          onChange={handleInputChange}
          className="w-full px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
          required
        >
          <option value="">Cinsiyet seçin</option>
          <option value="male">Erkek</option>
          <option value="female">Kadın</option>
        </select>
        {errors.gender && (
          <p className="text-sm text-red-400">{errors.gender}</p>
        )}
      </div>

      {/* Şehir */}
      <div className="space-y-2">
        <label className="block text-sm font-medium text-gray-200">
          Şehir <span className="text-red-400">*</span>
        </label>
        <select
          name="city"
          value={formData.city}
          onChange={handleInputChange}
          className="w-full px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
          required
        >
          <option value="">Şehir seçin</option>
          {cityOptions.map(city => (
            <option key={city} value={city}>{city}</option>
          ))}
        </select>
        {errors.city && (
          <p className="text-sm text-red-400">{errors.city}</p>
        )}
      </div>

      {/* Referans Kodu - Gizli/Açılabilir */}
      <div className="w-full">
        <div className="flex items-center justify-center">
          <button
            type="button"
            onClick={() => setShowReferralInput(!showReferralInput)}
            className="flex items-center gap-2 text-sm text-primary hover:text-primary/80 transition-colors"
          >
            <Iconify icon="material-symbols:card-giftcard" size={18} />
            <span>Referans kodum var</span>
            <Iconify 
              icon="mdi:chevron-down" 
              size={16} 
              className={`transition-transform duration-300 ${showReferralInput ? 'rotate-180' : ''}`}
            />
          </button>
        </div>
        
        <div 
          className={`transition-all duration-300 ease-in-out overflow-hidden ${
            showReferralInput ? 'max-h-40 opacity-100 mt-3' : 'max-h-0 opacity-0'
          }`}
        >
          <div className="space-y-2 w-full">
            <div className="flex items-center justify-between">
              <label className="block text-sm font-medium text-gray-200">
                Referans Kodu
              </label>
              <button
                type="button"
                onClick={() => {
                  setShowReferralInput(false)
                  setFormData(prev => ({ ...prev, referralCode: '' }))
                }}
                className="text-xs text-white/50 hover:text-white/70 transition-colors"
              >
                Gizle
              </button>
            </div>
            <Input
              name="referralCode"
              type="text"
              placeholder="Referans kodunuzu girin"
              value={formData.referralCode}
              onChange={handleInputChange}
              hint="Referans kodu ile kayıt olursanız bonus kazanırsınız!"
            />
          </div>
        </div>
      </div>

      <div className="grid grid-cols-2 gap-3 pt-4">
        <Button
          type="button"
          variant="outline"
          className="flex-1"
          onClick={handlePrevStep}
          size="md"
          fullWidth
        >
          <Iconify icon="material-symbols:arrow-back" size={16} className="mr-2" />
          Geri
        </Button>
        
        <Button
          type="submit"
          variant="primary"
          size="md"
          fullWidth
          className="flex-1"
          disabled={isLoading}

        >
          {isLoading ? (
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 border-2 border-white/30 border-t-white rounded-full animate-spin" />
              Kayıt Tamamlanıyor...
            </div>
          ) : (
            'Kayıt Tamamla'
          )}
        </Button>
      </div>
    </motion.form>
  )

  return (
    <>
      <Modal 
        isOpen={isOpen} 
        onClose={handleClose}
        size="md"
        image={currentStep === 1 ? '/assets/images/register-banner.png' : undefined}
        mobileImage={currentStep === 1 ? '/assets/images/register-banner-mobile.png' : undefined}
        fullscreen
     >
        <div className="space-y-4">
          <motion.h2 
            className="text-2xl font-semibold text-white text-start w-full pr-8"
            key={currentStep}
            initial={{ opacity: 0, y: -10 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.3 }}
          >
            {currentStep === 1 ? 'Kayıt Ol' : 'Daha Fazla Bilgi'}
          </motion.h2>
          <AnimatePresence mode="wait">
            {currentStep === 1 ? renderStep1() : renderStep2()}
          </AnimatePresence>
        </div>
      </Modal>

      {/* Success Modal */}
      <Modal isOpen={showSuccessModal} onClose={handleSuccessModalClose}>
        <motion.div 
          className="p-6 text-center space-y-6"
          initial={{ opacity: 0, scale: 0.9 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ duration: 0.3 }}
        >
          {/* Success Icon */}
          <motion.div 
            className="flex justify-center"
            initial={{ scale: 0 }}
            animate={{ scale: 1 }}
            transition={{ 
              delay: 0.2,
              type: "spring" as const,
              stiffness: 200,
              damping: 15
            }}
          >
            <StatCard
              icon="material-symbols:mail-outline"
              size="lg"
              color="green"
            />
          </motion.div>

          <div className="space-y-2">
            <h3 className="text-lg font-medium text-white">
              Mail adresinize doğrulama kodu gönderildi
            </h3>
            <p className="text-sm text-white/70">
              Ayarlardan doğrulama yapmayı unutmayınız
            </p>
          </div>
          
          <Button 
            variant="primary" 
            size="md" 
            fullWidth 
            onClick={handleSuccessModalClose}
          >
            Tamam
          </Button>
        </motion.div>
      </Modal>
    </>
  )
}