'use client'

import React from 'react'
import { cn } from '@/lib/utils'

export interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: 'primary' | 'secondary' | 'outline' | 'ghost' | 'danger'
  size?: 'sm' | 'md' | 'lg' | 'xl'
  loading?: boolean
  leftIcon?: React.ReactNode
  rightIcon?: React.ReactNode
  fullWidth?: boolean
}

const Button = React.forwardRef<HTMLButtonElement, ButtonProps>(
  ({
    className,
    variant = 'primary',
    size = 'md',
    loading = false,
    leftIcon,
    rightIcon,
    fullWidth = false,
    children,
    disabled,
    ...props
  }, ref) => {
    const baseStyles = [
      'inline-flex items-center justify-center gap-2',
      'font-medium transition-all duration-300',
      'disabled:opacity-50 disabled:cursor-not-allowed disabled:pointer-events-none',
      'rounded-xl relative overflow-hidden',
      'cursor-pointer',
    ]

    const variants = {
      primary: [
        'bg-primary hover:bg-primary/90 text-white',
        'before:absolute before:inset-0 before:bg-gradient-to-r before:from-transparent before:via-white/20 before:to-transparent',
        'before:translate-x-[-100%] hover:before:translate-x-[120%] before:transition-transform before:duration-900',
      ],
      secondary: [
        'bg-base-2 hover:bg-base-3 text-white',
        'border border-base-3 hover:border-primary/30',
      ],
      outline: [
        '',
      ],
      ghost: [
        'text-gray-300 hover:text-white hover:bg-base-2',
        'bg-transparent',
      ],
      danger: [
        'bg-red-600 hover:bg-red-700 text-white',
        'shadow-lg hover:shadow-xl',
      ],
    }

    const sizes = {
      sm: 'px-7 py-1.5 text-[11px] h-8',
      md: 'px-5 py-2.5 text-sm h-10',
      lg: 'px-6 py-3 text-base h-12',
      xl: 'px-8 py-4 text-lg h-14',
    }

    const isDisabled = disabled || loading

    if (variant === 'outline') {
      return(
      <button
        className={cn(
          baseStyles,
          fullWidth && 'w-full',
          className,
        )}
        disabled={isDisabled}
        ref={ref}
        {...props}
      >

      {loading && (
          <div className="absolute inset-0 flex items-center justify-center bg-inherit">
            <div className="w-4 h-4 border-2 border-current border-t-transparent rounded-full animate-spin" />
          </div>
        )}


        <div className={cn(
          'flex items-center justify-center bg-gradient-to-r from-primary/40 via-primary/20 to-primary/10 p-[2px] text-white hover:bg-base-2',
           loading && 'opacity-0',
           size === 'sm' && 'p-[1px]',
           fullWidth && 'w-full',

        )}>
            <div className={cn(
                'flex items-center justify-center rounded-[10px] bg-base-1',
                sizes[size],
                size === 'sm' && 'rounded-[11px]',
                fullWidth && 'w-full',
                )}>
               <div className='z-10 flex items-center justify-center space-x-2'>
                {leftIcon && <span className='flex items-center justify-center pr-2'>{leftIcon}</span>}
                {children}
                {rightIcon && <span className="flex items-center justify-center pl-2">{rightIcon}</span>}
            </div>
                <div className='z-0 blur-xl absolute transform -translate-x-1/2 translate-y-1/2 left-1/2 h-15 w-15 bg-primary/70 rounded-full'></div>
            </div>
      </div>
      </button>
    )

     {

    }

    }
    else {
      return (

      <button
        className={cn(
          baseStyles,
          variants[variant],
          sizes[size],
          fullWidth && 'w-full',
          className,
        )}
        disabled={isDisabled}
        ref={ref}
        {...props}
      >

      {loading && (
          <div className="absolute inset-0 flex items-center justify-center bg-inherit">
            <div className="w-4 h-4 border-2 border-current border-t-transparent rounded-full animate-spin" />
          </div>
        )}

    
  <div className={cn(
          loading && 'opacity-0',
          'flex items-center justify-center',
        )}>
          {leftIcon && <span className="flex items-center justify-center pr-2">{leftIcon}</span>}
          {children}
          {rightIcon && <span className="flex items-center justify-center pl-2">{rightIcon}</span>}
  </div>
    
      </button>

      )
  
  }
  })
Button.displayName = 'Button'

export { Button }
export default Button