'use client'

import React from 'react'
import { cn } from '@/lib/utils'
import { Iconify } from './Iconify'

export interface CheckboxProps {
  id?: string
  name?: string
  checked?: boolean
  defaultChecked?: boolean
  disabled?: boolean
  required?: boolean
  label?: React.ReactNode
  description?: string
  error?: string
  size?: 'sm' | 'md' | 'lg'
  variant?: 'default' | 'primary' | 'success' | 'warning' | 'danger'
  className?: string
  labelClassName?: string
  onChange?: (checked: boolean, event: React.ChangeEvent<HTMLInputElement>) => void
  onFocus?: (event: React.FocusEvent<HTMLInputElement>) => void
  onBlur?: (event: React.FocusEvent<HTMLInputElement>) => void
}

export function Checkbox({
  id,
  name,
  checked,
  defaultChecked,
  disabled = false,
  required = false,
  label,
  description,
  error,
  size = 'md',
  variant = 'primary',
  className,
  labelClassName,
  onChange,
  onFocus,
  onBlur,
  ...props
}: CheckboxProps) {
  const checkboxId = id || name || `checkbox-${Math.random().toString(36).substr(2, 9)}`
  
  const handleChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    if (onChange) {
      onChange(event.target.checked, event)
    }
  }

  // Size classes
  const sizeClasses = {
    sm: 'w-4 h-4',
    md: 'w-5 h-5',
    lg: 'w-6 h-6'
  }

  // Variant classes
  const variantClasses = {
    default: {
      checkbox: 'border-gray-600 text-gray-400 focus:ring-gray-500/20',
      checked: 'bg-gray-500 border-gray-500',
      icon: 'text-white'
    },
    primary: {
      checkbox: 'border-base-3 text-primary focus:ring-primary/20',
      checked: 'bg-primary border-primary',
      icon: 'text-white'
    },
    success: {
      checkbox: 'border-green-600 text-green-500 focus:ring-green-500/20',
      checked: 'bg-green-500 border-green-500',
      icon: 'text-white'
    },
    warning: {
      checkbox: 'border-yellow-600 text-yellow-500 focus:ring-yellow-500/20',
      checked: 'bg-yellow-500 border-yellow-500',
      icon: 'text-white'
    },
    danger: {
      checkbox: 'border-red-600 text-red-500 focus:ring-red-500/20',
      checked: 'bg-red-500 border-red-500',
      icon: 'text-white'
    }
  }

  const currentVariant = variantClasses[variant]

  return (
    <div className={cn('flex flex-col', className)}>
      <div className="flex items-start gap-3">
        <div className="relative flex items-center">
          <input
            id={checkboxId}
            name={name}
            type="checkbox"
            checked={checked}
            defaultChecked={defaultChecked}
            disabled={disabled}
            required={required}
            className="sr-only"
            onChange={handleChange}
            onFocus={onFocus}
            onBlur={onBlur}
            {...props}
          />
          
          {/* Custom Checkbox */}
          <label
            htmlFor={checkboxId}
            className={cn(
              'relative flex items-center justify-center rounded border-2 bg-base-2 transition-all duration-200 cursor-pointer',
              sizeClasses[size],
              checked ? currentVariant.checked : currentVariant.checkbox,
              disabled && 'opacity-50 cursor-not-allowed',
              error && 'border-red-500',
              'hover:border-opacity-80 focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-offset-base-1'
            )}
          >
            {checked && (
              <Iconify 
                icon="material-symbols:check" 
                size={size === 'sm' ? 12 : size === 'md' ? 16 : 20}
                className={cn(currentVariant.icon, 'transition-all duration-200')}
              />
            )}
          </label>
        </div>

        {/* Label and Description */}
        {(label || description) && (
          <div className="flex-1 min-w-0">
            {label && (
              <label
                htmlFor={checkboxId}
                className={cn(
                  'block text-sm font-medium cursor-pointer',
                  disabled ? 'text-gray-500' : 'text-gray-300',
                  error ? 'text-red-400' : '',
                  labelClassName
                )}
              >
                {label}
                {required && <span className="text-red-400 ml-1">*</span>}
              </label>
            )}
            
            {description && (
              <p className={cn(
                'text-sm mt-1',
                disabled ? 'text-gray-600' : 'text-gray-400',
                error ? 'text-red-400' : ''
              )}>
                {description}
              </p>
            )}
          </div>
        )}
      </div>

      {/* Error Message */}
      {error && (
        <div className="mt-2 ml-8">
          <p className="text-sm text-red-400 flex items-center gap-1">
            <Iconify icon="material-symbols:error-outline" size={16} />
            {error}
          </p>
        </div>
      )}
    </div>
  )
}

export default Checkbox