'use client'

import React from 'react'
import { Iconify } from './Iconify'
import { cn } from '@/lib/utils'

export interface PaginationProps {
  /** Mevcut sayfa numarası (1-indexed) */
  currentPage: number
  /** Toplam sayfa sayısı */
  totalPages: number
  /** Sayfa değiştiğinde çağrılacak fonksiyon */
  onPageChange: (page: number) => void
  /** Mobilde gösterilecek maksimum sayfa numarası */
  maxMobilePages?: number
  /** Desktop'ta gösterilecek maksimum sayfa numarası */
  maxDesktopPages?: number
  /** Ek CSS sınıfları */
  className?: string
}

export function Pagination({
  currentPage,
  totalPages,
  onPageChange,
  maxMobilePages = 3,
  maxDesktopPages = 7,
  className
}: PaginationProps) {
  // Sayfa numaralarını hesapla
  const getPageNumbers = (maxPages: number) => {
    const pages: (number | string)[] = []
    
    if (totalPages <= maxPages) {
      // Tüm sayfalar gösterilebilir
      for (let i = 1; i <= totalPages; i++) {
        pages.push(i)
      }
    } else {
      // Ellipsis ile göster
      const sidePages = Math.floor((maxPages - 3) / 2) // İlk, son ve mevcut sayfa için yer ayır
      
      if (currentPage <= sidePages + 2) {
        // Başta
        for (let i = 1; i <= maxPages - 2; i++) {
          pages.push(i)
        }
        pages.push('...')
        pages.push(totalPages)
      } else if (currentPage >= totalPages - sidePages - 1) {
        // Sonda
        pages.push(1)
        pages.push('...')
        for (let i = totalPages - maxPages + 3; i <= totalPages; i++) {
          pages.push(i)
        }
      } else {
        // Ortada
        pages.push(1)
        pages.push('...')
        for (let i = currentPage - sidePages; i <= currentPage + sidePages; i++) {
          pages.push(i)
        }
        pages.push('...')
        pages.push(totalPages)
      }
    }
    
    return pages
  }

  const mobilePages = getPageNumbers(maxMobilePages)
  const desktopPages = getPageNumbers(maxDesktopPages)

  const handlePrevious = () => {
    if (currentPage > 1) {
      onPageChange(currentPage - 1)
    }
  }

  const handleNext = () => {
    if (currentPage < totalPages) {
      onPageChange(currentPage + 1)
    }
  }

  const handlePageClick = (page: number | string) => {
    if (typeof page === 'number') {
      onPageChange(page)
    }
  }

  if (totalPages <= 1) return null

  return (
    <div className={cn('flex items-center justify-center gap-1 md:gap-2', className)}>
      {/* Previous Button */}
      <button
        onClick={handlePrevious}
        disabled={currentPage === 1}
        className={cn(
          'flex items-center justify-center w-8 h-8 md:w-10 md:h-10 rounded-lg transition-all',
          'bg-base-2 border border-white/5',
          currentPage === 1
            ? 'opacity-50 cursor-not-allowed text-gray-600'
            : 'hover:bg-base-3 hover:border-primary/30 text-white'
        )}
        aria-label="Önceki sayfa"
      >
        <Iconify icon="mdi:chevron-left" size={20} />
      </button>

      {/* Page Numbers - Mobile */}
      <div className="flex md:hidden items-center gap-1">
        {mobilePages.map((page, index) => (
          <button
            key={`mobile-${index}`}
            onClick={() => handlePageClick(page)}
            disabled={page === '...'}
            className={cn(
              'flex items-center justify-center min-w-8 h-8 rounded-lg transition-all text-xs font-medium',
              page === currentPage
                ? 'bg-primary text-white shadow-lg shadow-primary/25'
                : page === '...'
                ? 'cursor-default text-gray-400'
                : 'bg-base-2 text-gray-400 hover:bg-base-3 hover:text-white border border-white/5'
            )}
          >
            {page}
          </button>
        ))}
      </div>

      {/* Page Numbers - Desktop */}
      <div className="hidden md:flex items-center gap-2">
        {desktopPages.map((page, index) => (
          <button
            key={`desktop-${index}`}
            onClick={() => handlePageClick(page)}
            disabled={page === '...'}
            className={cn(
              'flex items-center justify-center min-w-10 h-10 rounded-lg transition-all text-sm font-medium',
              page === currentPage
                ? 'bg-primary text-white shadow-lg shadow-primary/25'
                : page === '...'
                ? 'cursor-default text-gray-400'
                : 'bg-base-2 text-gray-400 hover:bg-base-3 hover:text-white border border-white/5'
            )}
          >
            {page}
          </button>
        ))}
      </div>

      {/* Next Button */}
      <button
        onClick={handleNext}
        disabled={currentPage === totalPages}
        className={cn(
          'flex items-center justify-center w-8 h-8 md:w-10 md:h-10 rounded-lg transition-all',
          'bg-base-2 border border-white/5',
          currentPage === totalPages
            ? 'opacity-50 cursor-not-allowed text-gray-600'
            : 'hover:bg-base-3 hover:border-primary/30 text-white'
        )}
        aria-label="Sonraki sayfa"
      >
        <Iconify icon="mdi:chevron-right" size={20} />
      </button>
    </div>
  )
}

export default Pagination
