'use client'

import React, { useState } from 'react'
import { cn } from '@/lib/utils'

interface PlatformOption {
  id: string
  label: string
  value: string
}

export interface PlatformSelectorProps {
  /** Platform seçenekleri */
  options?: PlatformOption[]
  /** Varsayılan seçili platform */
  defaultSelected?: string
  /** Platform değiştiğinde çağrılacak fonksiyon */
  onChange?: (platform: string) => void
  /** Component boyutu */
  size?: 'sm' | 'md' | 'lg'
  /** Ek CSS sınıfları */
  className?: string
}

const PlatformSelector: React.FC<PlatformSelectorProps> = ({
  options = [
    { id: 'android', label: 'Android', value: 'android' },
    { id: 'ios', label: 'İos', value: 'ios' },
    { id: 'desktop', label: 'Masaüstü', value: 'desktop' }
  ],
  defaultSelected = 'android',
  onChange,
  size = 'md',
  className
}) => {
  const [selectedPlatform, setSelectedPlatform] = useState(defaultSelected)

  const handlePlatformChange = (platform: string) => {
    setSelectedPlatform(platform)
    onChange?.(platform)
  }

  // Seçili platform'un index'ini bul
  const selectedIndex = options.findIndex(option => option.value === selectedPlatform)

  // Boyut ayarları
  const sizeClasses = {
    sm: {
      container: 'w-90 ',
      button: 'text-[11px]',
      height: 'h-9'
    },
    md: {
      container: 'w-90 ',
      button: 'text-sm',
      height: 'h-9'
    },
    lg: {
      container: 'w-96',
      button: 'text-base',
      height: 'h-10'
    }
  }

  const currentSize = sizeClasses[size]

  // Slider pozisyonunu hesapla - Daha basit yaklaşım
  const getSliderStyle = () => {
    const width = 100 / options.length
    const left = selectedIndex * width
    
    return {
      width: `${width}%`,
      left: `${left}%`,
    }
  }

  return (
    <div className='bg-base-3 p-1 rounded-xl'>
    <div 
      className={cn(
        'relative inline-flex items-center rounded-lg',
        currentSize.container,
        currentSize.height,
        className,
      )}
    >
      {/* Sliding Background */}
      <div
        className={cn('absolute bg-primary rounded-lg transition-all duration-300 ease-out',
            currentSize.height,
        )}
        style={getSliderStyle()}
      />

      {/* Platform Buttons */}
      {options.map((option) => (
        <button
          key={option.id}
          onClick={() => handlePlatformChange(option.value)}
          className={cn(
            'relative z-10 font-medium rounded-md transition-all duration-200 cursor-pointer',
            'flex-1 flex items-center justify-center',
            currentSize.button,
            currentSize.height,
            selectedPlatform === option.value
              ? 'text-white'
              : 'text-gray-400 hover:text-gray-200'
          )}
        >
          {option.label}
        </button>
      ))}
    </div>
    </div>
  )
}

export { PlatformSelector }
export default PlatformSelector