'use client'

import React, { useState } from 'react'
import { cn } from '@/lib/utils'

export interface ToggleSwitchProps {
  /** Toggle'ın varsayılan durumu */
  defaultChecked?: boolean
  /** Toggle durumu değiştiğinde çağrılacak fonksiyon */
  onChange?: (checked: boolean) => void
  /** Toggle'ın boyutu */
  size?: 'sm' | 'md' | 'lg'
  /** Devre dışı bırakma durumu */
  disabled?: boolean
  /** Toggle'ın renk teması */
  variant?: 'primary' | 'success' | 'warning' | 'danger'
  /** Ek CSS sınıfları */
  className?: string
  /** Toggle'ın ID'si */
  id?: string
  /** Label metni */
  label?: string
  /** Label pozisyonu */
  labelPosition?: 'left' | 'right'
}

const ToggleSwitch: React.FC<ToggleSwitchProps> = ({
  defaultChecked = false,
  onChange,
  size = 'md',
  disabled = false,
  variant = 'primary',
  className,
  id,
  label,
  labelPosition = 'left'
}) => {
  const [isChecked, setIsChecked] = useState(defaultChecked)

  const handleToggle = () => {
    if (disabled) return
    
    const newChecked = !isChecked
    setIsChecked(newChecked)
    onChange?.(newChecked)
  }

  const sizeClasses = {
    sm: {
      container: 'w-8 h-5',
      thumb: 'w-3 h-3',
      translate: isChecked ? 'translate-x-3' : 'translate-x-0.5'
    },
    md: {
      container: 'w-10 h-6',
      thumb: 'w-4 h-4',
      translate: isChecked ? 'translate-x-5' : 'translate-x-1'
    },
    lg: {
      container: 'w-12 h-7',
      thumb: 'w-5 h-5',
      translate: isChecked ? 'translate-x-5' : 'translate-x-1'
    }
  }

  const variantClasses = {
    primary: isChecked ? 'bg-gradient-to-b from-[#B5A9FF]  to-[#36246A]' : 'bg-gray-500',
    success: isChecked ? 'bg-green-500' : 'bg-gray-600',
    warning: isChecked ? 'bg-yellow-500' : 'bg-gray-600',
    danger: isChecked ? 'bg-red-500' : 'bg-gray-600'
  }

  const currentSize = sizeClasses[size]
  const currentVariant = variantClasses[variant]

  const toggleElement = (
    <button
      type="button"
      role="switch"
      aria-checked={isChecked}
      disabled={disabled}
      onClick={handleToggle}
      id={id}
      className={cn(
        'relative inline-flex items-center rounded-full transition-all duration-500 ease-in-out',
        currentSize.container,
        currentVariant,
        disabled && 'opacity-50 cursor-not-allowed',
        !disabled && 'cursor-pointer',
        className
      )}
    >
      {/* Thumb (Moving Circle) */}
      <span
        className={cn(
          'inline-block  rounded-full bg-white shadow-lg transition-transform duration-500 ease-in-out',
          currentSize.thumb,
          currentSize.translate,
          disabled && 'shadow-sm'
        )}
      />
      
      {/* Screen reader text */}
      <span className="sr-only">
        {label ? `Toggle ${label}` : 'Toggle switch'}
      </span>
    </button>
  )

  // If no label, return just the toggle
  if (!label) {
    return toggleElement
  }

  // With label
  return (
    <div className={cn(
      'flex items-center gap-3 border',
      labelPosition === 'right' && 'flex-row-reverse'
     
    )}>
      {toggleElement}
      <label
        htmlFor={id}
        className={cn(
          'text-sm font-medium text-gray-300 cursor-pointer select-none',
          disabled && 'opacity-50 cursor-not-allowed'
        )}
      >
        {label}
      </label>
    </div>
  )
}

export { ToggleSwitch }
export default ToggleSwitch