/**
 * useAsync Hook
 * SOLID Principle: SRP - Async operations with loading/error states
 * SoC: Separate async logic from components
 */

'use client'

import { useState, useCallback, useEffect } from 'react'
import { logError, getUserFriendlyErrorMessage } from '@/lib/errors'

export interface UseAsyncState<T> {
  data: T | null
  loading: boolean
  error: string | null
  isSuccess: boolean
  isError: boolean
}

export interface UseAsyncOptions<T> {
  initialData?: T | null
  onSuccess?: (data: T) => void
  onError?: (error: string) => void
  autoFetch?: boolean
}

export function useAsync<T>(
  asyncFunction: () => Promise<T>,
  options: UseAsyncOptions<T> = {}
) {
  const {
    initialData = null,
    onSuccess,
    onError,
    autoFetch = false
  } = options

  const [state, setState] = useState<UseAsyncState<T>>({
    data: initialData,
    loading: autoFetch,
    error: null,
    isSuccess: false,
    isError: false,
  })

  const execute = useCallback(async () => {
    setState(prev => ({
      ...prev,
      loading: true,
      error: null,
      isError: false,
    }))

    try {
      const data = await asyncFunction()
      
      setState({
        data,
        loading: false,
        error: null,
        isSuccess: true,
        isError: false,
      })

      onSuccess?.(data)
      return data
    } catch (error) {
      const errorMessage = getUserFriendlyErrorMessage(error)
      logError(error, 'useAsync')

      setState({
        data: null,
        loading: false,
        error: errorMessage,
        isSuccess: false,
        isError: true,
      })

      onError?.(errorMessage)
      throw error
    }
  }, [asyncFunction, onSuccess, onError])

  const reset = useCallback(() => {
    setState({
      data: initialData,
      loading: false,
      error: null,
      isSuccess: false,
      isError: false,
    })
  }, [initialData])

  const retry = useCallback(() => {
    return execute()
  }, [execute])

  // Auto-fetch on mount if enabled
  useEffect(() => {
    if (autoFetch) {
      execute()
    }
  }, [autoFetch, execute])

  return {
    ...state,
    execute,
    reset,
    retry,
  }
}
