/**
 * useVerification Hook
 * SOLID Principle: SRP - Tek sorumluluk: Doğrulama işlemleri
 * SoC: Business logic'i component'ten ayır
 */

'use client'

import { useState } from 'react'
import { validateVerificationCode } from '@/lib/validation'

interface UseVerificationOptions {
  onSuccess?: () => void
  onError?: (error: string) => void
}

export function useVerification({ onSuccess, onError }: UseVerificationOptions = {}) {
  const [isLoading, setIsLoading] = useState(false)
  const [error, setError] = useState('')
  const [code, setCode] = useState('')

  const sendCode = async (target: string): Promise<boolean> => {
    setIsLoading(true)
    setError('')
    
    try {
      // Simulate API call
      console.log('Sending verification code to:', target)
      await new Promise(resolve => setTimeout(resolve, 1000))
      return true
    } catch (err) {
      const errorMessage = 'Kod gönderilirken bir hata oluştu'
      setError(errorMessage)
      onError?.(errorMessage)
      return false
    } finally {
      setIsLoading(false)
    }
  }

  const verifyCode = async (inputCode?: string): Promise<boolean> => {
    const codeToVerify = inputCode || code
    
    const validation = validateVerificationCode(codeToVerify)
    if (!validation.isValid) {
      setError(validation.error || '')
      onError?.(validation.error || '')
      return false
    }

    setIsLoading(true)
    setError('')
    
    try {
      // Simulate API call
      console.log('Verifying code:', codeToVerify)
      await new Promise(resolve => setTimeout(resolve, 1000))
      
      setCode('')
      onSuccess?.()
      return true
    } catch (err) {
      const errorMessage = 'Kod doğrulanamadı'
      setError(errorMessage)
      onError?.(errorMessage)
      return false
    } finally {
      setIsLoading(false)
    }
  }

  const resetCode = () => {
    setCode('')
    setError('')
  }

  return {
    code,
    setCode,
    error,
    setError,
    isLoading,
    sendCode,
    verifyCode,
    resetCode,
  }
}
