/**
 * useWithdrawForm Hook
 * SOLID Principle: SRP - Tek sorumluluk: Çekim formu state yönetimi
 * SoC: Form state logic'i component'ten ayır
 */

'use client'

import { useState } from 'react'
import { validateAmount, validateIBAN } from '@/lib/validation'

export interface WithdrawFormData {
  accountName?: string
  iban?: string
  walletAddress?: string
  amount: string
}

interface UseWithdrawFormOptions {
  balance: string
  onSubmit?: (data: WithdrawFormData) => void
}

export function useWithdrawForm({ balance, onSubmit }: UseWithdrawFormOptions) {
  const [formData, setFormData] = useState<WithdrawFormData>({
    accountName: '',
    iban: '',
    walletAddress: '',
    amount: ''
  })
  const [errors, setErrors] = useState<Partial<Record<keyof WithdrawFormData, string>>>({})

  const updateField = <K extends keyof WithdrawFormData>(
    field: K,
    value: WithdrawFormData[K]
  ) => {
    setFormData(prev => ({ ...prev, [field]: value }))
    // Clear error when field is updated
    if (errors[field]) {
      setErrors(prev => {
        const newErrors = { ...prev }
        delete newErrors[field]
        return newErrors
      })
    }
  }

  const setAmount = (value: string) => {
    // Sadece sayı ve nokta kabul et
    const numericValue = value.replace(/[^0-9.]/g, '')
    updateField('amount', numericValue)
  }

  const useAllBalance = () => {
    const numericBalance = balance.replace(/[₺,]/g, '').replace(',', '.')
    updateField('amount', numericBalance)
  }

  const validateForm = (type: 'bank' | 'crypto'): boolean => {
    const newErrors: Partial<Record<keyof WithdrawFormData, string>> = {}

    // Amount validation
    const amountValidation = validateAmount(formData.amount, balance)
    if (!amountValidation.isValid) {
      newErrors.amount = amountValidation.error
    }

    if (type === 'bank') {
      // Bank specific validations
      if (!formData.accountName?.trim()) {
        newErrors.accountName = 'Hesap adı gereklidir'
      }

      const ibanValidation = validateIBAN(formData.iban || '')
      if (!ibanValidation.isValid) {
        newErrors.iban = ibanValidation.error
      }
    } else {
      // Crypto specific validations
      if (!formData.walletAddress?.trim()) {
        newErrors.walletAddress = 'Cüzdan adresi gereklidir'
      }
    }

    setErrors(newErrors)
    return Object.keys(newErrors).length === 0
  }

  const handleSubmit = (type: 'bank' | 'crypto') => {
    if (validateForm(type)) {
      onSubmit?.(formData)
    }
  }

  const resetForm = () => {
    setFormData({
      accountName: '',
      iban: '',
      walletAddress: '',
      amount: ''
    })
    setErrors({})
  }

  return {
    formData,
    errors,
    updateField,
    setAmount,
    useAllBalance,
    validateForm,
    handleSubmit,
    resetForm,
  }
}
