/**
 * Error Handling Utilities
 * SOLID Principle: SRP - Error handling logic
 */

export class AppError extends Error {
  constructor(
    message: string,
    public code?: string,
    public statusCode?: number
  ) {
    super(message)
    this.name = 'AppError'
  }
}

export class ValidationError extends AppError {
  constructor(message: string, public field?: string) {
    super(message, 'VALIDATION_ERROR', 400)
    this.name = 'ValidationError'
  }
}

export class NetworkError extends AppError {
  constructor(message: string = 'Bağlantı hatası oluştu') {
    super(message, 'NETWORK_ERROR', 0)
    this.name = 'NetworkError'
  }
}

export class AuthError extends AppError {
  constructor(message: string = 'Oturum süreniz doldu') {
    super(message, 'AUTH_ERROR', 401)
    this.name = 'AuthError'
  }
}

export class NotFoundError extends AppError {
  constructor(message: string = 'Kayıt bulunamadı') {
    super(message, 'NOT_FOUND', 404)
    this.name = 'NotFoundError'
  }
}

/**
 * Error handler - API hatalarını işle
 */
export function handleApiError(error: unknown): AppError {
  // Network hatası
  if (!navigator.onLine) {
    return new NetworkError('İnternet bağlantınızı kontrol edin')
  }

  // AppError instance
  if (error instanceof AppError) {
    return error
  }

  // HTTP Response error
  if (error instanceof Response) {
    switch (error.status) {
      case 400:
        return new ValidationError('Geçersiz istek')
      case 401:
        return new AuthError()
      case 404:
        return new NotFoundError()
      case 500:
        return new AppError('Sunucu hatası', 'SERVER_ERROR', 500)
      default:
        return new AppError('Bir hata oluştu', 'UNKNOWN_ERROR', error.status)
    }
  }

  // Generic error
  if (error instanceof Error) {
    return new AppError(error.message, 'GENERIC_ERROR')
  }

  // Unknown error
  return new AppError('Beklenmeyen bir hata oluştu', 'UNKNOWN_ERROR')
}

/**
 * User-friendly error message generator
 */
export function getUserFriendlyErrorMessage(error: unknown): string {
  const appError = handleApiError(error)
  
  // Custom messages based on error type
  switch (appError.code) {
    case 'NETWORK_ERROR':
      return 'İnternet bağlantınızı kontrol edin ve tekrar deneyin'
    case 'AUTH_ERROR':
      return 'Oturumunuz sonlandı. Lütfen tekrar giriş yapın'
    case 'VALIDATION_ERROR':
      return 'Lütfen formu doğru şekilde doldurun'
    case 'NOT_FOUND':
      return 'Aradığınız sayfa bulunamadı'
    case 'SERVER_ERROR':
      return 'Sunucu hatası. Lütfen daha sonra tekrar deneyin'
    default:
      return appError.message || 'Bir hata oluştu. Lütfen tekrar deneyin'
  }
}

/**
 * Log error (production'da error tracking service'e gönderilecek)
 */
export function logError(error: unknown, context?: string) {
  const appError = handleApiError(error)
  
  // Development'ta console'a yaz
  if (process.env.NODE_ENV === 'development') {
    console.error(`[${context || 'Error'}]:`, {
      message: appError.message,
      code: appError.code,
      statusCode: appError.statusCode,
      stack: appError.stack,
    })
  }
  
  // Production'da error tracking service'e gönder
  // Örnek: Sentry.captureException(appError)
}
