/**
 * Type Guards - Runtime type checking
 * SOLID Principle: SRP - Type validation
 */

import type { Task } from '@/app/(dashboard)/_components/TaskTable'
import type { OfferCardData } from '@/data/offers'

/**
 * Primitive type guards
 */
export function isString(value: unknown): value is string {
  return typeof value === 'string'
}

export function isNumber(value: unknown): value is number {
  return typeof value === 'number' && !isNaN(value)
}

export function isBoolean(value: unknown): value is boolean {
  return typeof value === 'boolean'
}

export function isArray<T>(value: unknown): value is T[] {
  return Array.isArray(value)
}

export function isObject(value: unknown): value is Record<string, unknown> {
  return typeof value === 'object' && value !== null && !Array.isArray(value)
}

/**
 * Check if value is not null or undefined
 */
export function isDefined<T>(value: T | null | undefined): value is T {
  return value !== null && value !== undefined
}

/**
 * Check if string is not empty
 */
export function isNonEmptyString(value: unknown): value is string {
  return isString(value) && value.trim().length > 0
}

/**
 * Task type guard
 */
export function isTask(value: unknown): value is Task {
  if (!isObject(value)) return false
  
  return (
    isNonEmptyString(value.id) &&
    isNonEmptyString(value.logo) &&
    isNonEmptyString(value.title) &&
    isNonEmptyString(value.description) &&
    isNonEmptyString(value.amount) &&
    isNonEmptyString(value.status) &&
    isNonEmptyString(value.date)
  )
}

/**
 * Task array type guard
 */
export function isTaskArray(value: unknown): value is Task[] {
  if (!isArray(value)) return false
  return value.every(isTask)
}

/**
 * OfferCardData type guard
 */
export function isOfferCardData(value: unknown): value is OfferCardData {
  if (!isObject(value)) return false
  
  return (
    isNonEmptyString(value.id) &&
    isNonEmptyString(value.title) &&
    isNonEmptyString(value.subtitle) &&
    isNonEmptyString(value.amount) &&
    isNonEmptyString(value.image) &&
    isNonEmptyString(value.imageAlt) &&
    (value.rating === undefined || isNumber(value.rating))
  )
}

/**
 * OfferCardData array type guard
 */
export function isOfferCardDataArray(value: unknown): value is OfferCardData[] {
  if (!isArray(value)) return false
  return value.every(isOfferCardData)
}

/**
 * Validate and sanitize data with fallback
 */
export function validateWithFallback<T>(
  data: unknown,
  validator: (value: unknown) => value is T,
  fallback: T
): T {
  return validator(data) ? data : fallback
}

/**
 * Safe array getter with type checking
 */
export function safeArrayGet<T>(
  array: unknown,
  validator: (value: unknown) => value is T[]
): T[] {
  return validator(array) ? array : []
}

/**
 * Safe object property getter
 */
export function safeGet<T>(
  obj: unknown,
  key: string,
  fallback: T
): T {
  if (!isObject(obj)) return fallback
  const value = obj[key]
  return isDefined(value) ? value as T : fallback
}

/**
 * Email format check
 */
export function isValidEmail(value: unknown): value is string {
  if (!isString(value)) return false
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
  return emailRegex.test(value)
}

/**
 * Phone format check
 */
export function isValidPhone(value: unknown): value is string {
  if (!isString(value)) return false
  const phoneRegex = /^(\+90|0)?[0-9]{10}$/
  return phoneRegex.test(value.replace(/\s/g, ''))
}

/**
 * IBAN format check
 */
export function isValidIBAN(value: unknown): value is string {
  if (!isString(value)) return false
  const ibanRegex = /^TR\d{24}$/
  return ibanRegex.test(value.replace(/\s/g, '').toUpperCase())
}

/**
 * Amount validation
 */
export function isValidAmount(value: unknown): value is string {
  if (!isString(value)) return false
  const amountRegex = /^\d+(\.\d{1,2})?$/
  return amountRegex.test(value) && parseFloat(value) > 0
}
