/**
 * Validation Utilities
 * SOLID Principle: SRP - Tek sorumluluk: Form validation
 * SoC: Business logic'i presentation'dan ayır
 */

import { VALIDATION_RULES, ERROR_MESSAGES } from '@/constants'

export const validateEmail = (email: string): { isValid: boolean; error?: string } => {
  if (!email) {
    return { isValid: false, error: ERROR_MESSAGES.EMAIL_REQUIRED }
  }
  
  if (!VALIDATION_RULES.EMAIL.PATTERN.test(email)) {
    return { isValid: false, error: ERROR_MESSAGES.EMAIL_INVALID }
  }
  
  return { isValid: true }
}

export const validatePhone = (phone: string): { isValid: boolean; error?: string } => {
  if (!phone) {
    return { isValid: false, error: ERROR_MESSAGES.PHONE_REQUIRED }
  }
  
  const cleanPhone = phone.replace(/\s/g, '')
  if (!VALIDATION_RULES.PHONE.PATTERN.test(cleanPhone)) {
    return { isValid: false, error: ERROR_MESSAGES.PHONE_INVALID }
  }
  
  return { isValid: true }
}

export const validatePassword = (password: string): { isValid: boolean; error?: string } => {
  if (!password) {
    return { isValid: false, error: ERROR_MESSAGES.PASSWORD_REQUIRED }
  }
  
  if (password.length < VALIDATION_RULES.PASSWORD.MIN_LENGTH) {
    return { isValid: false, error: ERROR_MESSAGES.PASSWORD_WEAK }
  }
  
  if (!VALIDATION_RULES.PASSWORD.PATTERN.test(password)) {
    return { isValid: false, error: ERROR_MESSAGES.PASSWORD_WEAK }
  }
  
  return { isValid: true }
}

export const validatePasswordMatch = (
  password: string,
  confirmPassword: string
): { isValid: boolean; error?: string } => {
  if (password !== confirmPassword) {
    return { isValid: false, error: ERROR_MESSAGES.PASSWORD_MISMATCH }
  }
  
  return { isValid: true }
}

export const validateVerificationCode = (code: string): { isValid: boolean; error?: string } => {
  if (!code || code.length !== VALIDATION_RULES.VERIFICATION_CODE.LENGTH) {
    return { isValid: false, error: ERROR_MESSAGES.VERIFICATION_CODE_INVALID }
  }
  
  if (!VALIDATION_RULES.VERIFICATION_CODE.PATTERN.test(code)) {
    return { isValid: false, error: ERROR_MESSAGES.VERIFICATION_CODE_INVALID }
  }
  
  return { isValid: true }
}

export const validateIBAN = (iban: string): { isValid: boolean; error?: string } => {
  if (!iban) {
    return { isValid: false, error: ERROR_MESSAGES.REQUIRED }
  }
  
  const cleanIBAN = iban.replace(/\s/g, '').toUpperCase()
  if (!VALIDATION_RULES.IBAN.PATTERN.test(cleanIBAN)) {
    return { isValid: false, error: ERROR_MESSAGES.IBAN_INVALID }
  }
  
  return { isValid: true }
}

export const validateAmount = (amount: string, balance?: string): { isValid: boolean; error?: string } => {
  if (!amount) {
    return { isValid: false, error: ERROR_MESSAGES.REQUIRED }
  }
  
  const numericAmount = parseFloat(amount)
  
  if (isNaN(numericAmount) || numericAmount <= 0) {
    return { isValid: false, error: ERROR_MESSAGES.AMOUNT_INVALID }
  }
  
  if (balance) {
    const numericBalance = parseFloat(balance.replace(/[₺,]/g, ''))
    if (numericAmount > numericBalance) {
      return { isValid: false, error: 'Tutar bakiyeden fazla olamaz' }
    }
  }
  
  return { isValid: true }
}

export const validateUsername = (username: string): { isValid: boolean; error?: string } => {
  if (!username) {
    return { isValid: false, error: ERROR_MESSAGES.REQUIRED }
  }
  
  if (username.length < VALIDATION_RULES.USERNAME.MIN_LENGTH || 
      username.length > VALIDATION_RULES.USERNAME.MAX_LENGTH) {
    return { isValid: false, error: ERROR_MESSAGES.USERNAME_INVALID }
  }
  
  if (!VALIDATION_RULES.USERNAME.PATTERN.test(username)) {
    return { isValid: false, error: ERROR_MESSAGES.USERNAME_INVALID }
  }
  
  return { isValid: true }
}
