<?php

namespace LaraZeus\Qr\Facades;

use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Get;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Facade;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\HtmlString;
use LaraZeus\QrCode\Generator;

class Qr extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return 'qr';
    }

    public static function getDefaultOptions(array $options = []): array
    {
        return array_merge([
            'size' => '300',
            'type' => 'png',
            'margin' => '1',
            'color' => 'rgba(74, 74, 74, 1)',
            'back_color' => 'rgba(252, 252, 252, 1)',
            'style' => 'square',
            'hasGradient' => false,
            'gradient_form' => 'rgb(69, 179, 157)',
            'gradient_to' => 'rgb(241, 148, 138)',
            'gradient_type' => 'vertical',
            'hasEyeColor' => false,
            'eye_color_inner' => 'rgb(241, 148, 138)',
            'eye_color_outer' => 'rgb(69, 179, 157)',
            'eye_style' => 'square',
            'correction' => 'H',
            'percentage' => '.2',
            'uploadOptions' => [
                'disk' => 'public',
                'directory' => null,
            ],
        ], $options);
    }

    public static function getFormSchema(
        string $statePath,
        string $optionsStatePath,
        ?string $defaultUrl = 'https://',
        bool $showUrl = true,
        array $uploadOptions = []
    ): array {
        return [
            TextInput::make($statePath)
                ->live(onBlur: true)
                ->label('URL Giriniz')
                ->formatStateUsing(fn ($state) => $state ?? $defaultUrl)
                ->visible($showUrl),

            Grid::make()
                ->schema([
                    Section::make()
                        ->id('main-card')
                        ->columns(['sm' => 2])
                        ->columnSpan(['sm' => 2, 'lg' => 1])
                        ->statePath($optionsStatePath)
                        ->schema([
                            Hidden::make('type')->default('png'),
                            Select::make('correction')
                                ->live()
                                ->default('H')
                                ->label(__('QR Yoğunluğu'))
                                ->selectablePlaceholder(false)
                                ->columnSpan('full')
                                ->options([
                                    'L' => '7%',
                                    'M' => '15%',
                                    'Q' => '25%',
                                    'H' => '30%',
                                ]),

                            TextInput::make('size')
                                ->live()
                                ->default(300)
                                ->numeric()
                                ->label(__('Boyut')),

                            Select::make('margin')
                                ->live()
                                ->default(1)
                                ->label(__('Boşluk'))
                                ->selectablePlaceholder(false)
                                ->options([
                                    '0' => '0',
                                    '1' => '1',
                                    '3' => '3',
                                    '7' => '7',
                                    '9' => '9',
                                ]),

                            ColorPicker::make('color')
                                ->live()
                                ->default('rgba(74, 74, 74, 1)')
                                ->label(__('Renk'))
                                ->rgba(),

                            ColorPicker::make('back_color')
                                ->live()
                                ->default('rgba(252, 252, 252, 1)')
                                ->label(__('Arkaplan Rengi'))
                                ->rgba(),

                            Select::make('style')
                                ->selectablePlaceholder(false)
                                ->live()
                                ->columnSpanFull()
                                ->label(__('Stil'))
                                ->default('square')
                                ->options([
                                    'square' => __('Kare'),
                                    'round' => __('Yuvarlak'),
                                    'dot' => __('Nokta'),
                                ]),

                            Toggle::make('hasGradient')
                                ->live()
                                ->inline()
                                ->default(false)
                                ->columnSpanFull()
                                ->reactive()
                                ->label(__('Degrade')),

                            Grid::make()
                                ->schema([
                                    ColorPicker::make('gradient_form')
                                        ->live()
                                        ->default('rgb(69, 179, 157)')
                                        ->label(__('Degrade Başlangıç'))
                                        ->rgb(),

                                    ColorPicker::make('gradient_to')
                                        ->live()
                                        ->default('rgb(241, 148, 138)')
                                        ->label(__('Degrade Bitiş'))
                                        ->rgb(),

                                    Select::make('gradient_type')
                                        ->selectablePlaceholder(false)
                                        ->columnSpanFull()
                                        ->default('vertical')
                                        ->live()
                                        ->label(__('Degrade Tipi'))
                                        ->options([
                                            'vertical' => __('Dikey'),
                                            'horizontal' => __('Yatay'),
                                            'diagonal' => __('Çapraz'),
                                            'inverse_diagonal' => __('Ters Çapraz'),
                                            'radial' => __('Radyal'),
                                        ]),
                                ])
                                ->columnSpan(['sm' => 2])
                                ->columns(['sm' => 2])
                                ->visible(fn (Get $get) => $get('hasGradient')),

                            Toggle::make('hasEyeColor')
                                ->live()
                                ->inline()
                                ->columnSpanFull()
                                ->default(false)
                                ->label(__('Kutucuk Ayarı')),

                            Grid::make()
                                ->schema([
                                    ColorPicker::make('eye_color_inner')
                                        ->live()
                                        ->default('rgb(241, 148, 138)')
                                        ->label(__('İç kutucuk rengi'))
                                        ->rgb(),

                                    ColorPicker::make('eye_color_outer')
                                        ->live()
                                        ->default('rgb(69, 179, 157)')
                                        ->label(__('Dış kutucuk rengi'))
                                        ->rgb(),

                                    Select::make('eye_style')
                                        ->columnSpanFull()
                                        ->selectablePlaceholder(false)
                                        ->live()
                                        ->default('square')
                                        ->label(__('Kutucuk Tipi'))
                                        ->options([
                                            'square' => __('Kare'),
                                            'circle' => __('Yuvarlak'),
                                        ]),
                                ])
                                ->columnSpan(['sm' => 2])
                                ->columns(['sm' => 2])
                                ->visible(fn (Get $get) => $get('hasEyeColor')),

                            FileUpload::make('logo')
                                ->live()
                                ->imageEditor()
                                ->columnSpanFull()
                                ->disk($uploadOptions['disk'] ?? 'public')
                                ->directory($uploadOptions['directory'] ?? null)
                                ->image(),

                            Select::make('percentage')
                                ->live()
                                ->default(.2)
                                ->label(__('Resim Boyutu'))
                                ->visible(fn (Get $get) => $get('logo'))
                                ->selectablePlaceholder(false)
                                ->columnSpan('full')
                                ->options([
                                    '.1' => 'S',
                                    '.2' => 'M',
                                    '.3' => 'L',
                                    '.4' => 'XL',
                                ]),
                        ]),

                    Placeholder::make('preview')
                        ->label(__('Önizleme'))
                        ->columns(['sm' => 2])
                        ->columnSpan(['sm' => 2, 'lg' => 1])
                        ->key('preview_placeholder')
                        ->content(fn (Get $get) => Qr::render(
                            data: $get($statePath),
                            options: $get($optionsStatePath),
                            statePath: $statePath,
                            optionsStatePath: $optionsStatePath
                        )),
                ]),
        ];
    }

    // @internal
    public static function output(?string $data = null, ?array $options = null): HtmlString
    {
        $maker = new Generator;
        $size = 0.2;

        $options = $options ?? Qr::getDefaultOptions();

        call_user_func_array(
            [$maker, 'color'],
            ColorManager::getColorAsArray($options, 'color')
        );

        call_user_func_array(
            [$maker, 'backgroundColor'],
            ColorManager::getColorAsArray($options, 'back_color')
        );

        $maker = $maker->size(filled($options['size']) ? $options['size'] : static::getDefaultOptions()['size']);

        if ($options['hasGradient']) {
            if (filled($options['gradient_to']) && filled($options['gradient_form'])) {
                $gradient_form = ColorManager::getColorAsArray($options, 'gradient_form');
                $gradient_to = ColorManager::getColorAsArray($options, 'gradient_to');

                $gradientOptions = array_merge($gradient_to, $gradient_form, [$options['gradient_type']]);
                call_user_func_array([$maker, 'gradient'], $gradientOptions);
            }
        }

        if ($options['hasEyeColor']) {
            if (filled($options['eye_color_inner']) && filled($options['eye_color_outer'])) {
                $eye_color_inner = ColorManager::getColorAsArray($options, 'eye_color_inner');
                $eye_color_outer = ColorManager::getColorAsArray($options, 'eye_color_outer');

                $eyeColorInnerOptions = array_merge([0], $eye_color_inner, $eye_color_outer);
                call_user_func_array([$maker, 'eyeColor'], $eyeColorInnerOptions);

                $eyeColorInnerOptions = array_merge([1], $eye_color_inner, $eye_color_outer);
                call_user_func_array([$maker, 'eyeColor'], $eyeColorInnerOptions);

                $eyeColorInnerOptions = array_merge([2], $eye_color_inner, $eye_color_outer);
                call_user_func_array([$maker, 'eyeColor'], $eyeColorInnerOptions);
            }
        }

        if (filled($options['margin'])) {
            $maker = $maker->margin($options['margin']);
        }

        if (filled($options['correction'])) {
            $maker = $maker->errorCorrection($options['correction']);
        }

        if (filled($options['percentage'])) {
            $size = ($options['percentage']);
        }

        if (filled($options['style'])) {
            $maker = $maker->style($options['style']);
        }

        if (filled($options['eye_style'])) {
            $maker = $maker->eye($options['eye_style']);
        }

        if (optional($options)['logo']) {
            reset($options['logo']);
            $logo = current($options['logo']);

            if ($logo instanceof UploadedFile && filled($logo->getPathName())) {
                $maker = $maker->merge($logo->getPathName(), $size, true);
            } else {
                $disk = optional($options)['uploadOptions']['disk'] ?? 'public';
                if (Storage::disk($disk)->exists($logo)) {
                    $maker = $maker->merge(
                        Storage::disk($disk)->url($logo),
                        $size,
                        true
                    );
                }
            }
        }

        return new HtmlString(
            $maker->format(optional($options)['type'] ?? 'png')
                ->generate((filled($data) ? $data : 'https://'))
                ->toHtml()
        );
    }

    public static function render(
        ?string $data = null,
        ?array $options = null,
        string $statePath = 'url',
        string $optionsStatePath = 'options',
        bool $downloadable = true
    ): HtmlString {
        return new HtmlString(
            view('zeus-qr::download', [
                'optionsStatePath' => $optionsStatePath,
                'statePath' => $statePath,
                'data' => $data,
                'options' => $options ?? Qr::getDefaultOptions(),
                'downloadable' => $downloadable,
            ])->render()
        );
    }
}
